// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "uart.h"
#include "utility.h"

#define UART_TX_BUF_RDY_BIT (LATCbits.LATC5)
#define UART_TX_BUF_NOT_RDY 1
#define UART_TX_BUF_RDY     0

#define UART_BUF_SIZE 64

uint8_t uart_rxbuf[UART_BUF_SIZE];
uint8_t uart_txbuf[UART_BUF_SIZE];
volatile uint8_t uart_rxbuf_rp = 0;
volatile uint8_t uart_rxbuf_wp = 0;
volatile uint8_t uart_txbuf_rp = 0;
volatile uint8_t uart_txbuf_wp = 0;

#define PRINT_BUF_SIZE 9 // hu32 max digit is 8
char uart_print_buff[PRINT_BUF_SIZE];

void uart_init(void)
{
    U1BRG = 7; // FCY = 64MHz, Baud Rate = 500kHz

    PIE4bits.U1RXIE = 1; // Set U1RX interrupt enable bit
    PIE4bits.U1TXIE = 0; // Set U1TX interrupt enable bit

    // U1RX, U1TX default priority is high

    // Bit 5 - TXEN: 1 Transmit is enabled
    // Bit 4 - RXEN: 1 Receiver is enabled
    U1CON0 = 0x30;

    // Bit 7 - ON: 1 Serial port enabled
    U1CON1 = 0x80;
}

uint8_t uart_recv(uint8_t* buff, const uint8_t size)
{
    uint8_t i = 0;
    while(uart_rxbuf_rp != uart_rxbuf_wp && i++ < size)
    {
        *buff++ = uart_rxbuf[uart_rxbuf_rp];
        if(++uart_rxbuf_rp >= UART_BUF_SIZE) uart_rxbuf_rp = 0;
    }
    return i;
}

uint8_t uart_send(const uint8_t* buff, const uint8_t size, const uint8_t sync)
{
    uint8_t i = 0;
    while(i < size)
    {
        uint8_t next_wp = (uart_txbuf_wp + 1);
        if(next_wp >= UART_BUF_SIZE) next_wp = 0;
        if(next_wp != uart_txbuf_rp)
        {
            uart_txbuf[uart_txbuf_wp] = buff[i++];
            uart_txbuf_wp = next_wp;
        }
        else
        {
            if(sync)
            {
                volatile uint16_t j;
                if(uart_txbuf_wp != uart_txbuf_rp)
                {
                    // TX1IF is set when TX1REG gets to empty.
                    // But if U1TXIE is cleared and U1TXREG is still empty after that,
                    // U1TXIE will not be set.
                    PIE4bits.U1TXIE = 1;
                }
                for(j = 0; j < 1000; j++) {} // wait
            }
            else
            {
                break; // If buffer is full, return.
            }
        }
    }
    if(uart_txbuf_wp != uart_txbuf_rp)
    {
        // U1TXIE is set when TX1REG gets to empty.
        // But if U1TXIE is cleared and U1TXREG is still empty after that,
        // TX1IF will not be set.
        PIE4bits.U1TXIE = 1;
    }
    return i;
}

void uart_putc(const char c)
{
    uart_send((const uint8_t*)&c, 1, 1);
}

void uart_putc_async(const char c)
{
    uart_send((const uint8_t*)&c, 1, 0);
}

void uart_print(const char* str)
{
    uint8_t i;
    for(i = 0; str[i]; i++) {}
    uart_send((const uint8_t*)str, i, 1);
}

void uart_print_endl(void)
{
    uart_send((const uint8_t*)"\r\n", 2, 1);
}

void uart_print_endl_async(void)
{
    uart_send((const uint8_t*)"\r\n", 2, 0);
}

void uart_print_bytes(const uint8_t* bytes, const uint8_t size)
{
    uint8_t i;
    for(i = 0; i < size; i++)
    {
        u8tohs(uart_print_buff, bytes[i]);
        uart_send((uint8_t*)uart_print_buff, 2, 1);
    }
}

void uart_print_hu32(const uint32_t v)
{
    uint8_t i;
    i = u32tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_hu24(const __uint24 v)
{
    uint8_t i;
    i = u24tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_hu16(const uint16_t v)
{
    uint8_t i;
    i = u16tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_hu12(const uint16_t v)
{
    uint8_t i;
    i = u12tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_hu8(const uint8_t v)
{
    uint8_t i;
    i = u8tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_hu8_async(const uint8_t v)
{
    uint8_t i;
    i = u8tohs(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 0);
}

void uart_print_u32(const uint32_t v)
{
    uint8_t i;
    i = u32tos(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_u24(const __uint24 v)
{
    uint8_t i;
    i = u24tos(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_u16(const uint16_t v)
{
    uint8_t i;
    i = u16tos(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_u8(const uint8_t v)
{
    uint8_t i;
    i = u8tos(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void uart_print_i8(const int8_t v)
{
    uint8_t i;
    i = i8tos(uart_print_buff, v);
    uart_send((uint8_t*)uart_print_buff, i, 1);
}

void __interrupt(irq(U1RX)) uart_rx_int_handler(void)
{
    uint8_t v = U1RXB;
    uint8_t next_wp = uart_rxbuf_wp + 1;
    if(next_wp >= UART_BUF_SIZE) next_wp = 0;
    if(next_wp != uart_rxbuf_rp)
    {
        uart_rxbuf[uart_rxbuf_wp] = v;
        uart_rxbuf_wp = next_wp;
    }
    PIR4bits.U1RXIF = 0;
}

void __interrupt(irq(U1TX)) uart_tx_int_handler(void)
{
    if(uart_txbuf_wp != uart_txbuf_rp)
    {
        U1TXB = uart_txbuf[uart_txbuf_rp];
        if(++uart_txbuf_rp >= UART_BUF_SIZE) uart_txbuf_rp = 0;
        UART_TX_BUF_RDY_BIT = UART_TX_BUF_NOT_RDY;
    }
    else
    {
        PIE4bits.U1TXIE = 0;
        UART_TX_BUF_RDY_BIT = UART_TX_BUF_RDY;
    }
    //PIR4bits.U1TXIF = 0;
}
