// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "tmr.h"

uint32_t tmr_tick = 0;

void tmr_init(void)
{
    //--------------------------------------------------------------------------
    // TMR0
    // Freq: 64MHz / 4 / 128 / 125 = 1kHz
    // Fosc / 4 / (pre_scaler * (TMR0H + 1))
    TMR0H = 124;

    // T0CON1
    // Bits 7:5 - T0CS[2:0]: 010 FOSC/4
    // Bit 4 - ASYNC: 0 The input to the TMR0 counter is synchronized to Fosc/4
    // Bits 3:0 - CKPS[3:0]: 0111 1:128
    // 0100 0111
    T0CON1 = 0x47;

    PIR3bits.TMR0IF = 0;
    PIE3bits.TMR0IE = 1;

    // T0CON0
    // Bit 7 - T0EN TMR0 Enable: 1 The module is enabled and operating
    T0CON0bits.EN = 1;
}

uint8_t tmr_get_tick8(uint8_t digit)
{
    return ((uint8_t*)&tmr_tick)[digit];
}

uint32_t tmr_get_tick(void)
{
    volatile uint32_t tick0, tick1;
    do
    {
        tick0 = tmr_tick;
        tick1 = tmr_tick;
    } while(tick0 != tick1);
    return tick0;
}

void __interrupt(irq(TMR0)) tmr0_int_handler(void)
{
    PIR3bits.TMR0IF = 0;
    tmr_tick++;
}