// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>
#include "spic.h"
#include "tmr.h"
#include "chgv.h"
#include "uart.h"
#include "scmd_if.h"
#include "utility.h"

//#define SPIC_DEBUG
#define SPIC_RX_BUF_SIZE (2 + SCMD_MAX_LEN)
#define SPIC_TX_BUF_SIZE (2 + SCMD_MAX_LEN)

#define SPIC_ST_IDLE            0
#define SPIC_ST_READ_LEN        1
#define SPIC_ST_READ_LEN_BYPASS 2
#define SPIC_ST_BYPASS_PAR_CS   3
#define SPIC_ST_READ_PAR_CS     4
#define SPIC_ST_WAIT_PROC       5
#define SPIC_ST_SEND_RESPONSE   6

uint8_t spic_rx_buf[SPIC_RX_BUF_SIZE];
uint8_t spic_tx_buf[SPIC_TX_BUF_SIZE];
uint8_t spic_state;
uint8_t spic_rx_buf_cnt;
uint8_t spic_tx_buf_cnt;
uint8_t spic_rx_id;
uint8_t spic_rx_len;
uint8_t spic_cmd_got;
uint8_t spic_res_ready;
uint8_t spic_res_len;

void spic_init(void)
{
    spic_state = SPIC_ST_IDLE;

    // Bit 2 - SSET: 1 Client SS_in is ignored and data is clocked on all SCK_in (as though SS = TRUE at all times)
    // Bit 1 - TXR: 1 TxFIFO data is required for a transfer
    // Bit 0 - RXR: 1 Data transfers are suspended when RxFIFO is full
    SPI1CON2 = 0x07;

    PIR3bits.SPI1RXIF = 0;
    PIE3bits.SPI1RXIE = 1;
    SPI1STATUSbits.CLRBF = 1; // Clear TX, RX buffer
    SPI1TXB = 0xff; // Default TX byte
    SPI1CON0bits.EN = 1;
}

void spic_reset(void)
{
    INTCON0bits.GIE = 0;
    SPI1CON0bits.EN = 0;
    SPI1STATUSbits.CLRBF = 1; // Clear TX, RX buffer
    spic_init();
    INTCON0bits.GIE = 1;
}

// Return value: -1 No command received, -2 Error, 0 >= Command parameter size
int8_t spic_get_cmd(uint8_t* cmd_id, void* param, const uint8_t param_len)
{
    if(spic_cmd_got == 0)
    {
        return -1;
    }
    spic_cmd_got = 0;
    
    if(spic_rx_buf_cnt < 3)
    {
        return -2; // invalid command
    }
    
    if(param_len < (spic_rx_buf_cnt - 4))
    {
        return -3; // buffer overflow
    }

    // RX: header, length, cmd_id, param, checksum
    // Return only param
    *cmd_id = spic_rx_buf[2];
    memcpy(param, spic_rx_buf + 3, spic_rx_buf_cnt - 4);        
    return (int8_t)(spic_rx_buf_cnt - 4);
}

int8_t spic_set_res(const uint8_t res_id, const void* param, const uint8_t param_len)
{
    if((param_len + 4) > SPIC_TX_BUF_SIZE)
    {
        return -1; // buffer overflow
    }
    
    spic_tx_buf[0] = (uint8_t)(spic_rx_buf[0] << 4) | DEV_ID;
    spic_tx_buf[1] =  2 + param_len; // cmd_id + param + checksum
    spic_tx_buf[2] =  res_id;
    memcpy(spic_tx_buf + 3, param, param_len);     
    spic_tx_buf[param_len + 3] = calc_checksum(spic_tx_buf, spic_tx_buf[1] + 1);
    spic_res_len = spic_tx_buf[1] + 2;
    spic_res_ready = 1;
    
    return 0;    
}

void __interrupt(irq(SPI1RX)) spic_rx_int_handler(void)
{
    uint8_t rx_byte = SPI1RXB;
    uint8_t tx_byte;
#ifdef SPIC_DEBUG
    uart_putc_async('S');
    uart_print_hu8_async(spic_state);
    uart_print_hu8_async(rx_byte);
#endif
    PIR3bits.SPI1RXIF = 0;

    switch(spic_state)
    {
        case SPIC_ST_IDLE:
        {
            if(rx_byte == (DEV_ID << 4))
            {
                spic_rx_buf_cnt = 0;
                spic_rx_buf[spic_rx_buf_cnt++] = rx_byte;
                tx_byte = 0xff;
                spic_state = SPIC_ST_READ_LEN;
            }
            else if((SCMD_CDEV_ID_MIN << 4) <= rx_byte && rx_byte <= (SCMD_CDEV_ID_MAX << 4))
            {
                spic_rx_buf_cnt = 0;
                spic_rx_buf[spic_rx_buf_cnt++] = rx_byte;
                tx_byte = rx_byte;
                spic_state = SPIC_ST_READ_LEN_BYPASS;
            }
            else
            {
                // Header is not detected. Reset the state
                tx_byte = rx_byte;
                spic_state = SPIC_ST_IDLE;
            }
        }
        break;
        case SPIC_ST_READ_LEN:
        {
            spic_rx_buf[spic_rx_buf_cnt++] = rx_byte;
            spic_rx_len = 2 + rx_byte;
            if(rx_byte > SCMD_MAX_LEN || rx_byte == 0)
            {
                spic_state = SPIC_ST_IDLE;
                tx_byte = 0xff;
            }
            else
            {
                spic_state = SPIC_ST_READ_PAR_CS;
                tx_byte = 0xff;
            }
        }
        break;
        case SPIC_ST_READ_LEN_BYPASS:
        {
            spic_rx_buf[spic_rx_buf_cnt++] = rx_byte;
            spic_rx_len = 2 + rx_byte;
            if(rx_byte > SCMD_MAX_LEN || rx_byte == 0)
            {
                spic_state = SPIC_ST_IDLE;
                tx_byte = rx_byte;
            }
            else
            {
                spic_state = SPIC_ST_BYPASS_PAR_CS;
                tx_byte = rx_byte;
            }
        }
        break;
        case SPIC_ST_BYPASS_PAR_CS:
        {
            if(++spic_rx_buf_cnt >= spic_rx_len)
            {
                spic_state = SPIC_ST_IDLE;
            }
            tx_byte = rx_byte;
        }
        break;
        case SPIC_ST_READ_PAR_CS:
        {
            uint8_t res;
            spic_rx_buf[spic_rx_buf_cnt++] = rx_byte;
            if(spic_rx_buf_cnt >= spic_rx_len)
            {
                uint8_t calc_sum = calc_checksum(spic_rx_buf, spic_rx_len - 1);
                if(calc_sum != spic_rx_buf[spic_rx_len - 1])
                {
                    res = SCMD_RES_CHKSUM_ERR;
                    spic_set_res(SCMD_RES_CHKSUM_ERR, &res, 1);
                    spic_tx_buf_cnt = 0;
                    spic_state = SPIC_ST_SEND_RESPONSE;
                }
                else if(spic_rx_buf[2] > SCMD_ID_LAST_CMD)
                {
                    res = SCMD_RES_UNKN_CMD;
                    spic_set_res(SCMD_RES_CHKSUM_ERR, &res, 1);
                    spic_tx_buf_cnt = 0;
                    spic_state = SPIC_ST_SEND_RESPONSE;
                }
                else
                {
                    spic_cmd_got = 1;
                    spic_res_ready = 0;
                    spic_tx_buf_cnt = 0;
                    spic_state = SPIC_ST_WAIT_PROC;
                }
            }
            tx_byte = 0xff;
        }
        break;
        case SPIC_ST_WAIT_PROC:
        {
            if(spic_res_ready)
            {
                spic_res_ready = 0;
                tx_byte = spic_tx_buf[spic_tx_buf_cnt++];
                if(spic_tx_buf_cnt >= spic_res_len)
                {
                    // Sending was finished (irregular state)
                    spic_state = SPIC_ST_IDLE;
                }
                else
                {
                    spic_state = SPIC_ST_SEND_RESPONSE;
                }
            }
            else
            {
                tx_byte = 0xff;
            }
        }
        break;
        case SPIC_ST_SEND_RESPONSE:
        {
            tx_byte = spic_tx_buf[spic_tx_buf_cnt++];
            if(spic_tx_buf_cnt >= spic_res_len)
            {
                spic_state = SPIC_ST_IDLE;
            }
        }
        break;
    }
    SPI1TXB = tx_byte;
#ifdef SPIC_DEBUG
    uart_print_hu8_async(spic_state);
    uart_print_hu8_async(tx_byte);
    uart_print_endl_async();
#endif
}
