// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "opa.h"
#include "chgv.h"

void opa_set_mode(uint8_t mode);

void opa_init(void)
{
    opa_set_mode(OPA_MD_DISABLE);

    // Bits 6:4 - GSEL[2:0]: 111 1R 15R
    // Bit 3 - RESON: 1 Internal Resistor Ladder is enabled
    // Bits 2:0 - NSS[2:0]: 111 VSS
    OPA1CON1 = 0x7f;

    // Bits 6:4 - NCH[2:0]: 001 Internal Resistor Ladder (GSEL)
    // Bits 2:0 - PCH[2:0]: 010 OPAxIN+ (PSS)
    OPA1CON2 = 0x12;

    // Bits 7:6 - FMS[1:0]: 10 Operational Amplifier Output Pin (OPAxOUT)
    // Bits 1:0 - PSS[1:0]: 00 OPAxIN0+
    OPA1CON3 = 0x80;

    if(OPA_OFFSET > 0)
    {
        OPA1OFFSET = OPA_OFFSET;
    }
}

void opa_set_mode(uint8_t mode)
{
    switch(mode)
    {
    case OPA_MD_Q_CHG:
        // Bit 7 - EN: 0 Operational amplifier is disabled and consumes no active power
        // Bit 5 - CPON: 1 OPA Charge Pump on
        // Bit 3 - UG: 0 Inverting input is connected to designated OPAxIN- pin.
        // Bits 1:0 - SOC[1:0]: 00 Basic Operation; Operational amplifier configuration with user-defined or unity-gain feedback
        OPA1CON0 = 0x20;
        // Bits 6:4 - GSEL[2:0]: 111 1R 15R
        // Bit 3 - RESON: 1 Internal Resistor Ladder is enabled
        // Bits 2:0 - NSS[2:0]: 111 VSS
        OPA1CON1 = 0x7f;
        // Bits 6:4 - NCH[2:0]: 001 Internal Resistor Ladder (GSEL)
        // Bits 2:0 - PCH[2:0]: 010 OPAxIN+ (PSS)
        OPA1CON2 = 0x12;
        // Bits 7:6 - FMS[1:0]: 10 Operational Amplifier Output Pin (OPAxOUT)
        // Bits 1:0 - PSS[1:0]: 00 OPAxIN0+
        OPA1CON3 = 0x80;
        OPA1CON0bits.EN = 1;
        break;
    case OPA_MD_N_CHG:
        // Bit 7 - EN: 0 Operational amplifier is disabled and consumes no active power
        // Bit 5 - CPON: 1 OPA Charge Pump on
        // Bit 3 - UG: 0 Inverting input is connected to designated OPAxIN- pin.
        // Bits 1:0 - SOC[1:0]: 00 Basic Operation; Operational amplifier configuration with user-defined or unity-gain feedback
        // Bits 6:4 - GSEL[2:0]: 111 1R 15R
        // Bit 3 - RESON: 1 Internal Resistor Ladder is enabled
        // Bits 2:0 - NSS[2:0]: 111 VSS
        OPA1CON1 = 0x7f;
        // Bits 6:4 - NCH[2:0]: 001 Internal Resistor Ladder (GSEL)
        // Bits 2:0 - PCH[2:0]: 010 OPAxIN+ (PSS)
        OPA1CON2 = 0x12;
        OPA1CON0 = 0x20;
        // Bits 7:6 - FMS[1:0]: 10 Operational Amplifier Output Pin (OPAxOUT)
        // Bits 1:0 - PSS[1:0]: 01 OPAxIN1+
        OPA1CON3 = 0x81;
        OPA1CON0bits.EN = 1;
        break;
    case OPA_MD_OFF_CAL:
        // Bit 7 - EN: 0 Operational amplifier is disabled and consumes no active power
        // Bit 5 - CPON: 1 OPA Charge Pump on
        // Bit 3 - UG: 1 Operational amplifier output is connected to inverting input,
        // Bits 1:0 - SOC[1:0]: 00 Basic Operation; Operational amplifier configuration with user-defined or unity-gain feedback
        OPA1CON0 = 0x28;
        // Bits 6:4 - GSEL[2:0]: 111 1R 15R
        // Bit 3 - RESON: 1 Internal Resistor Ladder is enabled
        // Bits 2:0 - NSS[2:0]: 111 VSS
        OPA1CON1 = 0x7f;
        // Bits 6:4 - NCH[2:0]: 001 Internal Resistor Ladder (GSEL)
        // Bits 2:0 - PCH[2:0]: 101 DAC2_OUT
        OPA1CON2 = 0x15;
        // Bits 7:6 - FMS[1:0]: 10 Operational Amplifier Output Pin (OPAxOUT)
        // Bits 1:0 - PSS[1:0]: 01 OPAxIN1+
        OPA1CON3 = 0x80;
        OPA1CON0bits.EN = 1;
        break;
    default: //OPA_MD_DISABLE
        OPA1CON0bits.EN = 0;
        break;
    }
}
