// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// PIC18F15Q41 Configuration Bit Settings

// 'C' source line config statements

// CONFIG1
#pragma config FEXTOSC = OFF    // External Oscillator Selection (Oscillator not enabled)
#pragma config RSTOSC = HFINTOSC_64MHZ// Reset Oscillator Selection (HFINTOSC with HFFRQ = 64 MHz and CDIV = 1:1)

// CONFIG2
#pragma config CLKOUTEN = OFF   // Clock out Enable bit (CLKOUT function is disabled)
#pragma config PR1WAY = ON      // PRLOCKED One-Way Set Enable bit (PRLOCKED bit can be cleared and set only once)
#pragma config CSWEN = ON       // Clock Switch Enable bit (Writing to NOSC and NDIV is allowed)
#pragma config FCMEN = ON       // Fail-Safe Clock Monitor Enable bit (Fail-Safe Clock Monitor enabled)
#pragma config FCMENP = ON      // Fail-Safe Clock Monitor - Primary XTAL Enable bit (Fail-Safe Clock Monitor enabled; timer will flag FSCMP bit and OSFIF interrupt on EXTOSC failure.)
#pragma config FCMENS = ON      // Fail-Safe Clock Monitor - Secondary XTAL Enable bit (Fail-Safe Clock Monitor enabled; timer will flag FSCMP bit and OSFIF interrupt on SOSC failure.)

// CONFIG3
#pragma config MCLRE = EXTMCLR  // MCLR Enable bit (If LVP = 0, MCLR pin is MCLR; If LVP = 1, RE3 pin function is MCLR )
#pragma config PWRTS = PWRT_64  // Power-up timer selection bits (PWRT set at 64ms)
#pragma config MVECEN = ON      // Multi-vector enable bit (Multi-vector enabled, Vector table used for interrupts)
#pragma config IVT1WAY = ON     // IVTLOCK bit One-way set enable bit (IVTLOCKED bit can be cleared and set only once)
#pragma config LPBOREN = OFF    // Low Power BOR Enable bit (Low-Power BOR disabled)
#pragma config BOREN = SBORDIS  // Brown-out Reset Enable bits (Brown-out Reset enabled , SBOREN bit is ignored)

// CONFIG4
#pragma config BORV = VBOR_1P9  // Brown-out Reset Voltage Selection bits (Brown-out Reset Voltage (VBOR) set to 1.9V)
#pragma config ZCD = OFF        // ZCD Disable bit (ZCD module is disabled. ZCD can be enabled by setting the ZCDSEN bit of ZCDCON)
#pragma config PPS1WAY = ON     // PPSLOCK bit One-Way Set Enable bit (PPSLOCKED bit can be cleared and set only once; PPS registers remain locked after one clear/set cycle)
#pragma config STVREN = ON      // Stack Full/Underflow Reset Enable bit (Stack full/underflow will cause Reset)
#pragma config LVP = OFF        // Low Voltage Programming Enable bit (HV on MCLR/VPP must be used for programming)
#pragma config XINST = OFF      // Extended Instruction Set Enable bit (Extended Instruction Set and Indexed Addressing Mode disabled)

// CONFIG5
#pragma config WDTCPS = WDTCPS_31// WDT Period selection bits (Divider ratio 1:65536; software control of WDTPS)
#pragma config WDTE = OFF       // WDT operating mode (WDT Disabled; SWDTEN is ignored)

// CONFIG6
#pragma config WDTCWS = WDTCWS_7// WDT Window Select bits (window always open (100%); software control; keyed access not required)
#pragma config WDTCCS = SC      // WDT input clock selector (Software Control)

// CONFIG7
#pragma config BBSIZE = BBSIZE_512// Boot Block Size selection bits (Boot Block size is 512 words)
#pragma config BBEN = OFF       // Boot Block enable bit (Boot block disabled)
#pragma config SAFEN = OFF      // Storage Area Flash enable bit (SAF disabled)
#pragma config DEBUG = OFF      // Background Debugger (Background Debugger disabled)

// CONFIG8
#pragma config WRTB = OFF       // Boot Block Write Protection bit (Boot Block not Write protected)
#pragma config WRTC = OFF       // Configuration Register Write Protection bit (Configuration registers not Write protected)
#pragma config WRTD = OFF       // Data EEPROM Write Protection bit (Data EEPROM not Write protected)
#pragma config WRTSAF = OFF     // SAF Write protection bit (SAF not Write Protected)
#pragma config WRTAPP = OFF     // Application Block write protection bit (Application Block not write protected)

// CONFIG9
#pragma config CP = OFF         // PFM and Data EEPROM Code Protection bit (PFM and Data EEPROM code protection disabled)

// #pragma config statements should precede project file includes.
// Use project enums instead of #define for ON and OFF.

#include <xc.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include "uart.h"
#include "utility.h"
#include "adc.h"
#include "chgv.h"
#include "dcmd.h"
#include "dac.h"
#include "fvr.h"
#include "led.h"
#include "opa.h"
#include "pwm.h"
#include "tmr.h"
#include "spic.h"
#include "scmd_if.h"

#define _XTAL_FREQ    64000000
#define SCMD_BUF_SIZE 32

#define Q_CHG_DET_SEL LATAbits.LATA4
#define N_CHG_DET_SEL LATAbits.LATA5

void process_charge_init(void);
void process_charge_init_mes(void);
void process_charge_chk(void);
void process_charge_dsc(void);
void process_charge_dsc_mes(void);
void process_charge_n_chg(void);
void process_charge_n_vol_mes(void);
void process_charge_q_chg(void);
void process_charge_q_vol_mes(void);
void process_charge_err(void);
void process_charge_n_chg_vol_tst(void);
void process_charge_q_chg_vol_tst(void);
void process_charge_n_chg_tst(void);
void process_charge_n_vol_mes_tst(void);
void process_charge_q_chg_tst(void);
void process_charge_q_vol_mes_tst(void);
void process_charge_dsc_tst(void);
void process_charge_off_cal(void);
void process_charge_vol_mes_tst(void);

typedef struct _control_flag_t_tag
{
    uint8_t start_proc : 1;
    uint8_t print_status : 1;
} control_flag_t;

typedef struct _state_transition_entry
{
    uint8_t state;
    void (*func)(void);
} state_transition_entry_t;

const state_transition_entry_t state_trans_tbl[] =
{
    {CHGV_ST_INIT, process_charge_init},
    {CHGV_ST_FIN, process_charge_init},
    {CHGV_ST_INIT_MES, process_charge_init_mes},
    {CHGV_ST_FIN_MES, process_charge_init_mes},
    {CHGV_ST_CHK, process_charge_chk},
    {CHGV_ST_DSC, process_charge_dsc},
    {CHGV_ST_DSC_MES, process_charge_dsc_mes},
    {CHGV_ST_N_CHG, process_charge_n_chg},
    {CHGV_ST_N_VOL_MES, process_charge_n_vol_mes},
    {CHGV_ST_Q_CHG, process_charge_q_chg},
    {CHGV_ST_Q_VOL_MES, process_charge_q_vol_mes},
    {CHGV_ST_ERR, process_charge_err},
    {CHGV_ST_N_CHG_VOL_TST, process_charge_n_chg_vol_tst},
    {CHGV_ST_Q_CHG_VOL_TST, process_charge_q_chg_vol_tst},
    {CHGV_ST_N_CHG_TST, process_charge_n_chg_tst},
    {CHGV_ST_N_VOL_MES_TST, process_charge_n_vol_mes_tst},
    {CHGV_ST_Q_CHG_TST, process_charge_q_chg_tst},
    {CHGV_ST_Q_VOL_MES_TST, process_charge_q_vol_mes_tst},
    {CHGV_ST_DSC_TST, process_charge_dsc_tst},
    {CHGV_ST_OFF_CAL, process_charge_off_cal},
    {CHGV_ST_VOL_MES_TST, process_charge_vol_mes_tst}
};
#define STATE_TRANS_TBL_SIZE (sizeof(state_trans_tbl) / sizeof(state_trans_tbl[0]))

uint8_t prev_state;
uint8_t state = CHGV_ST_INIT;
uint8_t err_code;
uint8_t scmd_buf[SCMD_BUF_SIZE];
uint8_t q_chg_det_count;
uint16_t open_bat_vol;
uint16_t chgdsc_bat_vol;
uint16_t chgdsc_current;
uint16_t prev_chgdsc_bat_vol;
#define ctrl_req (chgv_ctrl.request)
#define ctrl_rst (chgv_ctrl.result)

// Global variables initialized by zero.
__uint24 duration_count;
uint16_t no_chk_duration_count;
uint16_t max_open_bat_vol;
control_flag_t control_flag;
uint8_t print_wait_count;
uint8_t cur_dsc_tgt_step;
uint8_t suppress_get_st = 1;
uint8_t enable_print_st = 0;
uint8_t chk_vol_wait = 0;

void port_init(void)
{
    U1RXPPS = 0x0f; // RB7 (001 111)
    RC6PPS = 0x0A; // PWM1S1P1_OUT
    RC7PPS = 0x10; // UART1 TX1

    // PPS lock
    INTCON0bits.GIE = 0; //Suspend interrupts
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 1; //Set PPSLOCKED bit
    INTCON0bits.GIE = 1; //Restore interrupts

    LATA = 0x00;   // 0000 0000
    TRISA = 0xcb;  // 1100 1011 RA5,4,2 output
    ANSELA = 0xcf; // 1100 1111 RA5,4 digital IO

    LATB = 0x00;   // 0000 0000
    INLVLC = 0x7f; // 0111 1111 RB7 TTL input
    //TRISB = 0xff;  // 1111 1111
    ANSELB = 0x2f; // 0010 1111 RB7,6,4 digital IO

    LATC = 0xc1;   // 1100 0001 RC7-6,0 high
    TRISC = 0x3c;  // 0011 1100 RC7-6,1-0 output
    ANSELC = 0x1c; // 0001 1100 RC7-5,1,0: digital IO
}

void reset(void)
{
    adc_stop();
    dac_set_step(0);
    opa_set_mode(OPA_MD_DISABLE);
    Q_CHG_DET_SEL = 0;
    N_CHG_DET_SEL = 0;
    open_bat_vol = 0;
    chgdsc_bat_vol = 0;
    chgdsc_current = 0;
    led_set_mode(LED_MD_OFF, 0);
    state = CHGV_ST_INIT;
    err_code = 0;
}

int8_t start(void)
{
    if(state == CHGV_ST_INIT || state == CHGV_ST_INIT_MES || state == CHGV_ST_FIN ||
       state == CHGV_ST_FIN_MES || state == CHGV_ST_ERR)
    {
        control_flag.start_proc = 1;
        return 0;
    }
    else
    {
        return -1;
    }
}

int8_t exec_test(uint8_t test_no)
{
    int8_t rv = 0;

    state = CHGV_ST_INIT;
    adc_stop();
    dac_set_step(0);
    open_bat_vol = 0;
    chgdsc_bat_vol = 0;
    chgdsc_current = 0;
    print_wait_count = 1;
    err_code = 0;
    Q_CHG_DET_SEL = 0;
    N_CHG_DET_SEL = 0;
    switch(test_no)
    {
    case 0:
        // Exit from test state
        break;
    case 1:
        Q_CHG_DET_SEL = 0;
        N_CHG_DET_SEL = 1;
        adc_tgt_vol = N_CHG_TGT_VOL;
        opa_set_mode(OPA_MD_N_CHG);
        adc_start(ADC_MD_N_CHG_TST);
        state = CHGV_ST_N_CHG_VOL_TST;
        break;
    case 2:
        Q_CHG_DET_SEL = 1;
        N_CHG_DET_SEL = 0;
        adc_tgt_vol = Q_CHG_TGT_VOL;
        opa_set_mode(OPA_MD_Q_CHG);
        adc_start(ADC_MD_Q_CHG_TST);
        state = CHGV_ST_Q_CHG_VOL_TST;
        break;
    case 3:
        Q_CHG_DET_SEL = 0;
        N_CHG_DET_SEL = 1;
        adc_tgt_vol = N_CHG_TGT_VOL;
        opa_set_mode(OPA_MD_N_CHG);
        adc_start(ADC_MD_N_CHG);
        state = CHGV_ST_N_CHG_TST;
        break;
    case 4:
        Q_CHG_DET_SEL = 1;
        N_CHG_DET_SEL = 0;
        adc_tgt_vol = Q_CHG_TGT_VOL;
        opa_set_mode(OPA_MD_Q_CHG);
        adc_start(ADC_MD_Q_CHG);
        state = CHGV_ST_Q_CHG_TST;
        break;
    case 5:
        Q_CHG_DET_SEL = 1;
        N_CHG_DET_SEL = 0;
        opa_set_mode(OPA_MD_DISABLE);
        dac_set_step(DSC_TGT_STEP);
        adc_start(ADC_MD_DSC);
        state = CHGV_ST_DSC_TST;
        break;
    case 6: // OPA offset calibration
        Q_CHG_DET_SEL = 0;
        N_CHG_DET_SEL = 1;
        opa_set_mode(OPA_MD_N_CHG);
        adc_start(ADC_MD_OFF_CAL);
        state = CHGV_ST_OFF_CAL;
        break;
    case 7: // ADC value check
        Q_CHG_DET_SEL = 1;
        N_CHG_DET_SEL = 0;
        opa_set_mode(OPA_MD_DISABLE);
        adc_start(ADC_MD_OP_VOL);
        state = CHGV_ST_VOL_MES_TST;
        break;
    default:
        rv = -1;
        break;
    }

    return rv;
}

uint8_t check_adc_error(void)
{
    if(adc_result == ADC_RST_ERR)
    {
        return CHGV_ERR_OVER_CUR;
    }
    return CHGV_ERR_NONE;
}

uint8_t check_charge_error_state(void)
{
    if(open_bat_vol == 0x000)
    {
        return CHGV_ERR_SHORT;
    }
    else if(open_bat_vol == 0xfff)
    {
        return CHGV_ERR_OPEN;
    }
    return CHGV_ERR_NONE;
}

void process_charge_init(void)
{
    adc_stop();
    opa_set_mode(OPA_MD_DISABLE);
    dac_set_step(0);
    open_bat_vol = 0;
    chgdsc_bat_vol = 0;
    chgdsc_current = 0;
    prev_chgdsc_bat_vol = 0;
    Q_CHG_DET_SEL = 0;
    N_CHG_DET_SEL = 0;
    if(control_flag.start_proc)
    {
        control_flag.start_proc = 0;
        control_flag.print_status = 1;
        Q_CHG_DET_SEL = 0;
        N_CHG_DET_SEL = 1;
        print_wait_count = 1;
        chk_vol_wait = 10;
        state = CHGV_ST_CHK;
        // Set minimum discharge current to discharge the capacitor of BAT_VOL detection LPF. 
        dac_set_step(1);
        adc_start(ADC_MD_DSC); //adc_start(ADC_MD_OP_VOL2);
    }
    else if(IDLE_MES)
    {
        control_flag.start_proc = 0;
        control_flag.print_status = 1;
        Q_CHG_DET_SEL = 0;
        N_CHG_DET_SEL = 1;
        print_wait_count = 1;
        state = (state == CHGV_ST_INIT) ? CHGV_ST_INIT_MES: CHGV_ST_FIN_MES;
        adc_start(ADC_MD_OP_VOL2);
    }
}

void process_charge_init_mes(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        open_bat_vol = adc_val[0];
        chgdsc_bat_vol = chgdsc_current = 0;
        if(--print_wait_count == 0)
        {
            print_wait_count = (suppress_get_st) ? 10: 1;
            control_flag.print_status = 1;
        }
        if(control_flag.start_proc)
        {
            control_flag.start_proc = 0;
            control_flag.print_status = 1;
            Q_CHG_DET_SEL = 0;
            N_CHG_DET_SEL = 1;
            chk_vol_wait = 10;
            state = CHGV_ST_CHK;
        }
        adc_start(ADC_MD_OP_VOL2);
    }
}

void process_charge_chk(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // Open voltage check is performed after chk_vol_wait * 100ms
        if(--chk_vol_wait > 0)
        {
            adc_start(ADC_MD_OP_VOL2);
        }
        else
        {
            open_bat_vol = adc_val[0];
            chgdsc_bat_vol = chgdsc_current = 0;
            if(open_bat_vol < BAT_MIN_VOL)
            {
                control_flag.print_status = 1;
                state = CHGV_ST_ERR;
                err_code = CHGV_ERR_BAT_MIN_VOL;
                led_set_mode(LED_MD_N_BLINK, err_code);
            }
            else if(open_bat_vol > MIN_CHG_STA_VOL)
            {
                cur_dsc_tgt_step = DSC_TGT_STEP;
                dac_set_step(cur_dsc_tgt_step);
                led_set_mode(LED_MD_BLINK, 0);
                control_flag.print_status = 1;
                print_wait_count = 1;
                state = CHGV_ST_DSC;
                adc_start(ADC_MD_DSC);
            }
            else if(open_bat_vol < Q_CHG_MIN_VOL)
            {
                duration_count = N_CHG_TIME;
                adc_tgt_vol = N_CHG_TGT_VOL;
                led_set_mode(LED_MD_ON, 0);
                opa_set_mode(OPA_MD_N_CHG);
                control_flag.print_status = 1;
                print_wait_count = 1;
                state = CHGV_ST_N_CHG;
                adc_start(ADC_MD_N_CHG);
            }
            else
            {
                Q_CHG_DET_SEL = 1;
                N_CHG_DET_SEL = 0;
                duration_count = Q_CHG_TIME;
                control_flag.print_status = 1;
                adc_tgt_vol = Q_CHG_TGT_VOL;
                led_set_mode(LED_MD_ON, 0);
                opa_set_mode(OPA_MD_Q_CHG);
                prev_chgdsc_bat_vol = 0;
                no_chk_duration_count = Q_CHG_NO_CHK_DUR;
                max_open_bat_vol = 0;
                print_wait_count = 1;
                q_chg_det_count = 0;
                state = CHGV_ST_Q_CHG;
                adc_start(ADC_MD_Q_CHG);
            }
        }
    }
}

void process_charge_dsc(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // cur_dsc_tgt_step: 4mV step
        // dsc_cur[mA] = cur_dsc_tgt_step * 4mV / 0.5ohm = cur_dsc_tgt_step * 8
        chgdsc_current = (uint16_t)((uint16_t)cur_dsc_tgt_step << 3);
        chgdsc_bat_vol = adc_val[0];
        dac_set_step(0);
        state = CHGV_ST_DSC_MES;
        adc_start(ADC_MD_OP_VOL);
    }
}

void process_charge_dsc_mes(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        open_bat_vol = adc_val[0];
        if(chgdsc_bat_vol <= DSC_END_VOL)
        {
            if(cur_dsc_tgt_step > DSC_END_TGT_STEP)
            {
                // Continue discharge with lower current
                cur_dsc_tgt_step >>= 1;
                if(cur_dsc_tgt_step < DSC_END_TGT_STEP) cur_dsc_tgt_step = DSC_END_TGT_STEP;
                dac_set_step(cur_dsc_tgt_step);
                state = CHGV_ST_DSC;
                adc_start(ADC_MD_DSC);
            }
            else
            {
                // Discharge was finished
                if(DSC_ONLY)
                {
                    uart_print("Discharge only. Return to INIT\r\n"); // for debug
                    led_set_mode(LED_MD_OFF, 0);
                    state = CHGV_ST_INIT;
                }
                else
                {
                    Q_CHG_DET_SEL = 1;
                    N_CHG_DET_SEL = 0;
                    duration_count = Q_CHG_TIME;
                    adc_tgt_vol = Q_CHG_TGT_VOL;
                    led_set_mode(LED_MD_ON, 0);
                    opa_set_mode(OPA_MD_Q_CHG);
                    prev_chgdsc_bat_vol = 0;
                    no_chk_duration_count = Q_CHG_NO_CHK_DUR;
                    max_open_bat_vol = 0;
                    print_wait_count = 1;
                    q_chg_det_count = 0;
                    state = CHGV_ST_Q_CHG;
                    adc_start(ADC_MD_Q_CHG);
                }
            }
            control_flag.print_status = 1;
        }
        else
        {
            // continue discharge with the same current
            if(--print_wait_count == 0)
            {
                print_wait_count = (suppress_get_st) ? 10: 1;
                control_flag.print_status = 1;
            }
            dac_set_step(cur_dsc_tgt_step);
            state = CHGV_ST_DSC;
            adc_start(ADC_MD_DSC);
        }
    }
}

void process_charge_n_chg(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // chg_cur[mA] = adc_val[1] / 16 / 0.5ohm = adc_val[1] / 8
        chgdsc_current = (adc_val[1] >> 3);
        chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        open_bat_vol = 0;
        err_code = check_adc_error();
        if(err_code == CHGV_ERR_NONE)
        {
            // Measure open voltage
            state = CHGV_ST_N_VOL_MES;
            adc_start(ADC_MD_OP_VOL);
        }
        else
        {
            control_flag.print_status = 1;
            state = CHGV_ST_ERR;
            led_set_mode(LED_MD_N_BLINK, err_code);
        }
    }
}

void process_charge_n_vol_mes(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        open_bat_vol = adc_val[0];
        duration_count--;
        err_code = check_charge_error_state();
        if(err_code != CHGV_ERR_NONE)
        {
            control_flag.print_status = 1;
            state = CHGV_ST_ERR;
            led_set_mode(LED_MD_N_BLINK, err_code);
        }
        else if(adc_val[0] >= N_CHG_FIN_VOL)
        {
            // Battery voltage was recovered to 1.2V. Discharge once to re-activate the battery.
            adc_stop();
            opa_set_mode(OPA_MD_DISABLE);
            Q_CHG_DET_SEL = 0;
            N_CHG_DET_SEL = 0;
            open_bat_vol = 0;
            chgdsc_bat_vol = 0;
            chgdsc_current = 0;
            prev_chgdsc_bat_vol = 0;
            cur_dsc_tgt_step = DSC_TGT_STEP;
            dac_set_step(cur_dsc_tgt_step);
            led_set_mode(LED_MD_BLINK, 0);
            control_flag.print_status = 1;
            print_wait_count = 1;
            state = CHGV_ST_DSC;
            adc_start(ADC_MD_DSC);
        }
        else if(duration_count == 0)
        {
            // Charge limit duration passed.
            control_flag.print_status = 1;
            state = CHGV_ST_ERR;
            err_code = CHGV_ERR_CHG_DUR;
            led_set_mode(LED_MD_N_BLINK, err_code);
        }
        else
        {
            // Continue to charge
            if(--print_wait_count == 0)
            {
                print_wait_count = (suppress_get_st) ? 10: 1;
                control_flag.print_status = 1;
            }
            state = CHGV_ST_N_CHG;
            adc_start(ADC_MD_N_CHG);
        }
    }
}

void process_charge_q_chg(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // chg_cur[mA] = adc_val[1] / 16 / 0.1ohm = adc_val[1] * 10 / 16
        chgdsc_current = ((adc_val[1] * 10) >> 4);
        chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        open_bat_vol = 0;
        err_code = check_adc_error();
        if(err_code == CHGV_ERR_NONE)
        {
            // Measure open voltage
            state = CHGV_ST_Q_VOL_MES;
            adc_start(ADC_MD_OP_VOL);
        }
        else
        {
            control_flag.print_status = 1;
            state = CHGV_ST_ERR;
            led_set_mode(LED_MD_N_BLINK, err_code);
        }
    }
}

void process_charge_q_vol_mes(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        open_bat_vol = adc_val[0];
        prev_chgdsc_bat_vol = (prev_chgdsc_bat_vol + prev_chgdsc_bat_vol + prev_chgdsc_bat_vol +
                               prev_chgdsc_bat_vol + prev_chgdsc_bat_vol + prev_chgdsc_bat_vol +
                               prev_chgdsc_bat_vol + chgdsc_bat_vol) >> 3;

        duration_count--;
        if(no_chk_duration_count > 0)
        {
            no_chk_duration_count--;
        }

        err_code = check_charge_error_state();
        if(err_code != CHGV_ERR_NONE)
        {
            control_flag.print_status = 1;
            state = CHGV_ST_ERR;
            led_set_mode(LED_MD_N_BLINK, err_code);
        }
        else if(no_chk_duration_count > 0)
        {
            // Continue to charge
            if(--print_wait_count == 0)
            {
                print_wait_count = (suppress_get_st) ? 10: 1;
                control_flag.print_status = 1;
            }
            state = CHGV_ST_Q_CHG;
            adc_start(ADC_MD_Q_CHG);
        }
        else
        {
            if(open_bat_vol > max_open_bat_vol)
            {
                max_open_bat_vol = open_bat_vol;
                control_flag.print_status = 1;
            }
            if((max_open_bat_vol - open_bat_vol) >= Q_CHG_DET_VOL)
            {
                if(++q_chg_det_count >= 5)
                {
                    // Voltage down was detected. Stop charging.
                    led_set_mode(LED_MD_OFF, 0);
                    state = CHGV_ST_FIN;
                }
                control_flag.print_status = 1;
            }
            else if(duration_count == 0)
            {
                // Charge limit duration passed.
                control_flag.print_status = 1;
                state = CHGV_ST_ERR;
                err_code = CHGV_ERR_CHG_DUR;
                led_set_mode(LED_MD_N_BLINK, err_code);
            }
            else
            {
                // Continue to charge
                if(--print_wait_count == 0)
                {
                    print_wait_count = (suppress_get_st) ? 10: 1;
                    control_flag.print_status = 1;
                }
                q_chg_det_count = 0;
                state = CHGV_ST_Q_CHG;
                adc_start(ADC_MD_Q_CHG);
            }
        }
    }
}

void process_charge_err(void)
{
    adc_stop();
    opa_set_mode(OPA_MD_DISABLE);
    dac_set_step(0);
    open_bat_vol = 0;
    chgdsc_bat_vol = 0;
    chgdsc_current = 0;
    Q_CHG_DET_SEL = 0;
    N_CHG_DET_SEL = 0;
}

void process_charge_n_chg_vol_tst(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // ADC error check is disabled
        adc_result = ADC_RST_NONE;
        // chg_cur[mA] = adc_val[1] / 16 / 0.5ohm = adc_val[1] / 8
        //chgdsc_current = (adc_val[1] >> 3);
        //chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        chgdsc_current = adc_val[1];
        chgdsc_bat_vol = adc_val[0];
        open_bat_vol = 0;
        control_flag.print_status = 1;
    }
}

void process_charge_q_chg_vol_tst(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // ADC error check is disabled
        adc_result = ADC_RST_NONE;
        // chg_cur[mA] = adc_val[1] / 16 / 0.1ohm = adc_val[1] * 10 / 16
        //chgdsc_current = ((adc_val[1] * 10) >> 4);
        //chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        chgdsc_current = adc_val[1];
        chgdsc_bat_vol = adc_val[0];
        open_bat_vol = 0;
        control_flag.print_status = 1;
    }
}

void process_charge_n_chg_tst(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // ADC error check is disabled
        // chg_cur[mA] = adc_val[1] / 16 / 0.5ohm = adc_val[1] / 8
        chgdsc_current = (adc_val[1] >> 3);
        chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        open_bat_vol = 0;
        err_code = check_adc_error();

        // Ignore error
        // Measure open voltage
        state = CHGV_ST_N_VOL_MES_TST;
        adc_start(ADC_MD_OP_VOL);
    }
}

void process_charge_n_vol_mes_tst(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        open_bat_vol = adc_val[0];
        err_code = check_charge_error_state();
        // Ignore error

        if(--print_wait_count == 0)
        {
            print_wait_count = (suppress_get_st) ? 10: 1;
            control_flag.print_status = 1;
        }
        state = CHGV_ST_N_CHG_TST;
        adc_start(ADC_MD_N_CHG);
    }
}

void process_charge_q_chg_tst(void)
{
    if(adc_result != ADC_RST_NONE)
    {
        // ADC error check is disabled
        // chg_cur[mA] = adc_val[1] / 16 / 0.1ohm = adc_val[1] * 10 / 16
        chgdsc_current = ((adc_val[1] * 10) >> 4);
        chgdsc_bat_vol = adc_val[0] - (adc_val[1] >> 4);
        open_bat_vol = 0;
        err_code = check_adc_error();

        // Ignore error
        // Measure open voltage
        state = CHGV_ST_Q_VOL_MES_TST;
        adc_start(ADC_MD_OP_VOL);
    }
}

void process_charge_q_vol_mes_tst(void)
{
    if(adc_result == ADC_RST_CMP)
    {
        open_bat_vol = adc_val[0];
        err_code = check_charge_error_state();
        // Ignore error

        if(--print_wait_count == 0)
        {
            print_wait_count = (suppress_get_st) ? 10: 1;
            control_flag.print_status = 1;
        }
        state = CHGV_ST_Q_CHG_TST;
        adc_start(ADC_MD_Q_CHG);
    }
}

void process_charge_dsc_tst(void)
{
    if(adc_result == ADC_RST_CMP)
    {
        // cur_dsc_tgt_step: 4mV step
        // dsc_cur[mA] = cur_dsc_tgt_step * 4mV / 0.5ohm = cur_dsc_tgt_step * 8
        chgdsc_current = (uint16_t)((uint16_t)cur_dsc_tgt_step << 3);
        chgdsc_bat_vol = adc_val[0];
        control_flag.print_status = 1;
        adc_start(ADC_MD_DSC);
    }
}

void process_charge_off_cal(void)
{
    if(adc_result == ADC_RST_CMP)
    {
        adc_result = ADC_RST_NONE;
        chgdsc_bat_vol = adc_val[1];
        open_bat_vol = adc_val[0];
        control_flag.print_status = 1;
    }
}

void process_charge_vol_mes_tst(void)
{
    if(adc_result == ADC_RST_CMP)
    {
        chgdsc_bat_vol = adc_val[1];
        open_bat_vol = adc_val[0];
        control_flag.print_status = 1;
        adc_start(ADC_MD_OP_VOL);
    }
}

void write_status(void)
{
    if(chgv_status_cnt >= CHGV_STAUS_NUM)
    {
        // Discard the oldest status
        chgv_status_cnt--;
        if(++chgv_status_top >= CHGV_STAUS_NUM) chgv_status_top = 0;
    }

    uint8_t next = chgv_status_top + chgv_status_cnt;
    if(next >= CHGV_STAUS_NUM) next -= CHGV_STAUS_NUM;

    chgv_status[next].tick = tmr_get_tick();
    chgv_status[next].prev_state = prev_state;
    chgv_status[next].state = state;
    chgv_status[next].err_code = err_code;
    chgv_status[next].open_bat_vol = open_bat_vol;
    chgv_status[next].chgdsc_bat_vol = chgdsc_bat_vol;
    chgv_status[next].chgdsc_current = chgdsc_current;
    chgv_status[next].prev_chgdsc_bat_vol = prev_chgdsc_bat_vol;

    chgv_status_cnt++;
}

void process_charge(void)
{
    uint8_t i;

    uint8_t state0 = state;
    for(i = 0; i < STATE_TRANS_TBL_SIZE; i++)
    {
        if(state_trans_tbl[i].state == state)
        {
            state_trans_tbl[i].func();
            break;
        }
    }
    if(state != state0) prev_state = state0;
    if(control_flag.print_status)
    {
        write_status();
        control_flag.print_status = 0;
    }

    if(enable_print_st)
    {
        while(chgv_status_cnt > 0)
        {
            chgv_print_status(&chgv_status[chgv_status_top], 0);
            if(++chgv_status_top >= CHGV_STAUS_NUM) chgv_status_top = 0;
            chgv_status_cnt--;
        }
    }
}

void main(void)
{
    chgv_load_config();
    port_init();
    uart_init();
    fvr_init();
    tmr_init();
    adc_init();
    pwm_init();
    opa_init();
    dac_init();

    // Enable interrupt
    INTCON0 = 0x80;

    while(1)
    {
        dcmd_execute();
        process_charge();
        led_update();
    }
}

