// Copyright 2023 Tomoyuki Watanabe
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//     http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>
#include "i2c.h"
#include "tmr.h"
#include "chg_var.h"
#include "uart.h"

#define I2C_DEBUG

#define I2C_IDLE           0
#define I2C_READ_DEV_ADDR  1
#define I2C_READ_DATA      2
#define I2C_WRITE_DEV_ADDR 3
#define I2C_WRITE_REG      4
#define I2C_WRITE_DATA     5

uint8_t i2c_reg_addr = 0;
uint8_t i2c_flag_updated = 0;
uint8_t i2c_state = 0;

void i2cc_init(void)
{
    I2C1PIR = 0;
    // Bit 3 - ADRIE: 1 Enables Address Interrupt and Hold condition
    // Bit 2 - PCIE: Enables interrupt on the detection of a Stop condition
    //I2C1PIE = 0x0c;
    I2C1PIE = 0x08;
    I2C1ADR0 = I2C1ADR1 = I2C1ADR2 = I2C1ADR3 = (uint8_t)(DEV_ADDR << 1);
    I2C1CNTL  = 0xff;
    I2C1CNTH = 0;
    
    PIE7bits.I2C1EIE = 1;
    PIE7bits.I2C1IE = 1;
    PIE7bits.I2C1TXIE = 1;
    PIE7bits.I2C1RXIE = 1;
    
    I2C1CON0bits.EN = 1;
}

void i2cc_tx_byte(void)
{
    if(i2c_reg_addr < CHG_VAR_CFG_ADDR)
    {
        uint8_t b = chg_var_status[chg_var_status_rp].b[i2c_reg_addr];
        if(i2c_reg_addr == CHG_VAR_STA_END)
        {
            if(chg_var_status_rp != chg_var_status_wp)
            {
                chg_var_status_rp++;
                if(chg_var_status_rp >= CHG_VAR_STAUS_NUM) chg_var_status_rp = 0;
            }
        }
        i2c_reg_addr++;
        I2C1TXB = b;
    }
    else if(i2c_reg_addr < CHG_VAR_CTL_ADDR)
    {
        I2C1TXB = chg_var_config.b[i2c_reg_addr - CHG_VAR_CFG_ADDR];
        i2c_reg_addr++;
    }
    else if(i2c_reg_addr < CHG_VAR_SIZE)
    {
        I2C1TXB = chg_var_ctrl.b[i2c_reg_addr - CHG_VAR_CTL_ADDR];
        chg_var_ctrl.r.result = CHG_VAR_CTL_RST_BUSY;
        i2c_reg_addr++;
    }
    else
    {
        I2C1TXB = 0xff;
    }
}

void i2cc_rx_byte(void)
{
    if(i2c_reg_addr < CHG_VAR_CFG_ADDR)
    {
        chg_var_status[chg_var_status_rp].b[i2c_reg_addr] = I2C1RXB;
        i2c_reg_addr++;
    }
    else if(i2c_reg_addr < CHG_VAR_CTL_ADDR)
    {
        chg_var_config.b[i2c_reg_addr - CHG_VAR_CFG_ADDR] = I2C1RXB;
        i2c_reg_addr++;
    }
    else if(i2c_reg_addr < CHG_VAR_SIZE)
    {
        chg_var_ctrl.b[i2c_reg_addr - CHG_VAR_CTL_ADDR] = I2C1RXB;
        i2c_reg_addr++;
    }
    else
    {
        I2C1STAT1bits.CLRBF = 1;
    }
}

void i2cc_process(void)
{
#ifdef I2C_DEBUG
    uart_print_hu8_async(PIR7);
    uart_print_hu8_async(I2C1PIR);
    uart_print_hu8_async(I2C1ERR);
    uart_print_hu8_async(I2C1STAT0);
    uart_print_hu8_async(I2C1STAT1);
    uart_print_hu8_async(I2C1CON1);
    uart_print_hu8_async(I2C1CNTL);
    uart_print_hu8_async(i2c_reg_addr);
    uart_print_hu8_async(i2c_state);
    uart_putc_async('\r');
    uart_putc_async('\n');
#endif
    
    if(PIR7bits.I2C1EIF)
    {
        // I2C1EIF is ORed of I2C1ERR bits  
        I2C1ERR = 0;  
    }
    else if(I2C1PIRbits.ADRIF)
    {
        I2C1PIRbits.ADRIF = 0;
        if(I2C1STAT0bits.R)
        {
            I2C1CON1bits.ACKDT = (i2c_reg_addr < CHG_VAR_SIZE) ? 0 : 1;
            I2C1STAT1bits.CLRBF = 1;
            //if(I2C1STAT1bits.TXBE)
            //{
                i2cc_tx_byte();
            //}
            i2c_state = I2C_READ_DATA;
        }
        else
        {
            I2C1CON1bits.ACKDT = 0; // always ACK
            I2C1STAT1bits.CLRBF = 1;
            i2c_state = I2C_WRITE_REG;
        }
        I2C1ERR = 0;
        I2C1PIR = 0; // should clear PCIF
        I2C1PIEbits.PCIE = 1;
        I2C1CON0bits.CSTR = 0; // Resume operation
    }
    else if(PIR7bits.I2C1TXIF)
    {
        i2cc_tx_byte();
    }
    else if(PIR7bits.I2C1RXIF)
    {
        if(i2c_state == I2C_WRITE_REG)
        {
            uint8_t addr = I2C1RXB;
            if(addr < CHG_VAR_SIZE)
            {
                i2c_reg_addr = addr;
                i2c_state = I2C_WRITE_DATA;
            }
            else
            {
                I2C1CON1bits.ACKDT = 1;                        
            }
        }
        else
        {
            i2cc_rx_byte();
        }
    }
    /*else if(PIR7bits.I2C1TXIF || PIR7bits.I2C1RXIF)
    {
        switch(i2c_state)
        {
            case I2C_READ_DATA:
                if(I2C1STAT1bits.TXBE)
                {
                    i2cc_tx_byte();
                }
                break;
            case I2C_WRITE_REG:
                if(I2C1STAT1bits.RXBF)
                {
                    uint8_t addr = I2C1RXB;
                    if(addr < CHG_VAR_SIZE)
                    {
                        i2c_reg_addr = addr;
                        i2c_state = I2C_WRITE_DATA;
                    }
                    else
                    {
                        I2C1CON1bits.ACKDT = 1;                        
                    }
                }
                break;
            case I2C_WRITE_DATA:
                if(I2C1STAT1bits.RXBF)
                {
                    i2cc_rx_byte();
                }
                break;
            default:
                break;
        }
    }*/
    else if(I2C1PIRbits.PCIF)
    {
        //if(!I2C1STAT1bits.TXBE)
        //{
        //    if(0 < i2c_reg_addr) i2c_reg_addr--;
        //}
        I2C1STAT1bits.CLRBF = 1;
        I2C1CNTL  = 0xff; // dont' use ACKCNT
        i2c_state = I2C_IDLE;
        I2C1ERR = 0;
        I2C1PIR = 0;
        I2C1PIEbits.PCIE = 0;
        I2C1CON1bits.ACKDT = 0; // need to set ACK, but SMA is not set when host sends start condition...
        I2C1CON0bits.CSTR = 0; // Resume operation
    }
}

void __interrupt(irq(I2C1E)) i2cc_error_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('E');
#endif
    i2cc_process();
}

void __interrupt(irq(I2C1RX)) i2cc_rx_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('R');
#endif
    i2cc_process();
}

void __interrupt(irq(I2C1TX)) i2cc_tx_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('T');
#endif
    i2cc_process();
}

void __interrupt(irq(I2C1)) i2cc_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('I');
#endif
    i2cc_process();
}
