#include "uart.h"
#include "utility.h"
#include "adc.h"
#include "chgv.h"
#include "dcmd.h"
#include "dac.h"
#include "fvr.h"
#include "spic.h"
#include "led.h"
#include "opa.h"
#include "pwm.h"
#include "tmr.h"

#define DCMD_BUF_SIZE 32

#define DCMD_BATCAP "batcap"
#define DCMD_BATMAXVOL "batmaxvol"
#define DCMD_BATMINVOL "batminvol"
#define DCMD_DACSTEP "dacstep"
#define DCMD_DSCENDTGTSTEP "dscendtgtstep"
#define DCMD_DSCENDVOL "dscendvol"
#define DCMD_DSCONLY "dsconly"
#define DCMD_DSCONLY "dsconly"
#define DCMD_DSCTGTSTEP "dsctgtstep"
#define DCMD_ENPRINTST "enprintst"
#define DCMD_GET "get"
#define DCMD_IDLEMES "idlemes"
#define DCMD_IDLEMES "idlemes"
#define DCMD_LED "led"
#define DCMD_MINCHGSTAVOL "minchgstavol"
#define DCMD_NCHGFINVOL "nchgfinvol"
#define DCMD_NCHGTGTVOL "nchgtgtvol"
#define DCMD_NCHGTIME "nchgtime"
#define DCMD_OPAOFFSET "opaoffset"
#define DCMD_PCON "pcon"
#define DCMD_PRINTCFG "printcfg"
#define DCMD_PWMDUTY "pwmduty"
#define DCMD_QCHGDETVOL "qchgdetvol"
#define DCMD_QCHGMINVOL "qchgminvol"
#define DCMD_QCHGNOCHKDUR "qchgnochkdur"
#define DCMD_QCHGNOCHKVOL "qchgnochkvol"
#define DCMD_QCHGTGTVOL "qchgtgtvol"
#define DCMD_QCHGTIME "qchgtime"
#define DCMD_RESET "reset"
#define DCMD_SAVECFG "savecfg"
#define DCMD_SIGNATURE "signature"
#define DCMD_START "start"
#define DCMD_STATUS "status"
#define DCMD_SUPGETST "supgetst"
#define DCMD_TEST "test"

typedef struct _dbg_cmd_t
{
    const char* cmd;
    void (*exec)(char* args, void* param);
    void *param;
} dbg_cmd_t;

// Debug access to members in main.c
extern int8_t start(void);
extern void reset(void);
extern int8_t exec_test(uint8_t test_no);
extern uint8_t enable_print_st;
extern uint8_t suppress_get_st;
extern uint8_t prev_state;
extern uint8_t state;
extern uint8_t err_code;
extern void write_status(void);

// Debug access to members in spic.c
extern uint8_t spic_state;

char dcmd_buf[DCMD_BUF_SIZE];
uint8_t dcmd_buf_len;

void dcmd_set_u8(char* args, void* param);
void dcmd_set_u16(char* args, void* param);
void dcmd_set_u24(char* args, void* param);
void dcmd_batcap(char* args, void* param);
void dcmd_dacstep(char* args, void* param);
void dcmd_led(char* args, void* param);
void dcmd_pcon(char* args, void* param);
void dcmd_printcfg(char* args, void* param);
void dcmd_reset(char* args, void* param);
void dcmd_savecfg(char* args, void* param);
void dcmd_start(char* args, void* param);
void dcmd_status(char* args, void* param);
void dcmd_test(char* args, void* param);

const dbg_cmd_t dcmd_list[] =
{
    {DCMD_BATCAP,        dcmd_set_u16,  &BAT_CAP},
    {DCMD_BATMAXVOL,     dcmd_set_u16,  &BAT_MAX_VOL},
    {DCMD_BATMINVOL,     dcmd_set_u16,  &BAT_MIN_VOL},
    {DCMD_DACSTEP,       dcmd_dacstep,  NULL},
    {DCMD_DSCENDTGTSTEP, dcmd_set_u8,   &DSC_END_TGT_STEP},
    {DCMD_DSCENDVOL,     dcmd_set_u16,  &DSC_END_VOL},
    {DCMD_DSCONLY,       dcmd_set_u8,   &DSC_ONLY},
    {DCMD_DSCTGTSTEP,    dcmd_set_u8,   &DSC_TGT_STEP},
    {DCMD_ENPRINTST,     dcmd_set_u8,   &enable_print_st},
    {DCMD_IDLEMES,       dcmd_set_u8,   &IDLE_MES},
    {DCMD_LED,           dcmd_led,      NULL},
    {DCMD_MINCHGSTAVOL,  dcmd_set_u16,  &MIN_CHG_STA_VOL},
    {DCMD_NCHGFINVOL,    dcmd_set_u16,  &N_CHG_FIN_VOL},
    {DCMD_NCHGTGTVOL,    dcmd_set_u16,  &N_CHG_TGT_VOL},
    {DCMD_NCHGTIME,      dcmd_set_u24,  &N_CHG_TIME},
    {DCMD_OPAOFFSET,     dcmd_set_u8,   &OPA_OFFSET},
    {DCMD_PCON,          dcmd_pcon,     NULL},
    {DCMD_PRINTCFG,      dcmd_printcfg, NULL},
    {DCMD_PWMDUTY,       dcmd_set_u16,  &adc_tst_pwm_duty},
    {DCMD_QCHGDETVOL,    dcmd_set_u16,  &Q_CHG_DET_VOL},
    {DCMD_QCHGMINVOL,    dcmd_set_u16,  &Q_CHG_MIN_VOL},
    {DCMD_QCHGNOCHKDUR,  dcmd_set_u16,  &Q_CHG_NO_CHK_DUR},
    {DCMD_QCHGNOCHKVOL,  dcmd_set_u16,  &Q_CHG_NO_CHK_VOL},
    {DCMD_QCHGTGTVOL,    dcmd_set_u16,  &Q_CHG_NO_CHK_VOL},
    {DCMD_QCHGTIME,      dcmd_set_u24,  &Q_CHG_TIME},
    {DCMD_RESET,         dcmd_reset,    NULL},
    {DCMD_SAVECFG,       dcmd_savecfg,  NULL},
    {DCMD_START,         dcmd_start,    NULL},
    {DCMD_STATUS,        dcmd_status,   NULL},
    {DCMD_SUPGETST,      dcmd_set_u8,   &suppress_get_st},
    {DCMD_TEST,          dcmd_test,     NULL}
};
#define CMD_COUNT (uint8_t)(sizeof(dcmd_list) / sizeof(dcmd_list[0]))

void dcmd_set_u8(char* args, void* param)
{
    uint8_t i = 1;
    if(args[0] != 0)
    {
        i = stou8((uint8_t*)param, args);
    }
    uart_print((i > 0) ? "ok ": "ng ");
    uart_print_u8(*((uint8_t*)param)); uart_print_endl();
}

void dcmd_set_u16(char* args, void* param)
{
    uint8_t i = 1;
    if(args[0] != 0)
    {
        i = stou16((uint16_t*)param, args);
        if(i > 0 && param == &BAT_CAP)
        {
            chgv_cal_tgt_vol();
        }
    }
    uart_print((i > 0) ? "ok ": "ng ");
    uart_print_u16(*((uint16_t*)param)); uart_print_endl();
}

void dcmd_set_u24(char* args, void* param)
{
    uint8_t i = 1;
    if(args[0] != 0)
    {
        i = stou24((__uint24*)param, args);
    }
    uart_print((i > 0) ? "ok ": "ng ");
    uart_print_u24(*((__uint24*)param)); uart_print_endl();
}

void dcmd_dacstep(char* args, void* param)
{
    (void)param;
    uint8_t i, v;
    i = stou8(&v, args);
    if(i > 0)
    {
        dac_set_step(v);
        uart_print("ok "); uart_print_u8(v); uart_print_endl();
    }
    else
    {
        uart_print("ng"); uart_print_endl();
    }
}

void dcmd_led(char* args, void* param)
{
    (void)param;
    uint8_t i;
    uint8_t mode, bl_cnt;
    uint8_t err = 0;

    i = stou8(&mode, args);
    if(i == 0)
    {
        err = 1;
        goto error_exit;
    }
    i = stou8(&bl_cnt, args + i);
    if(i == 0)
    {
        err = 2;
        goto error_exit;
    }
    led_set_mode(mode, bl_cnt);
    uart_print("ok ");
    uart_print_hu8(led_mode); uart_print_hu8(led_blink_count);
    goto end;

error_exit:
    uart_print("ng "); uart_print_hu8(err);
end:
    uart_print_endl();
}

void dcmd_pcon(char* args, void* param)
{
    (void)args;
    (void)param;
    uart_print_hu8(PCON0);
    uart_print(" ");
    uart_print_hu8(PCON1);
    uart_print_endl();
    uart_print("ok"); uart_print_endl();
}

void dcmd_printcfg(char* args, void* param)
{
    (void)args;
    (void)param;
    chgv_print_config();
    uart_print("ok"); uart_print_endl();
}

void dcmd_reset(char* args, void* param)
{
    (void)args;
    (void)param;
    reset();
    uart_print("ok "); uart_print_endl();
}

void dcmd_savecfg(char* args, void* param)
{
    (void)args;
    (void)param;
    int8_t rv = chgv_save_config();
    uart_print((rv == 0) ? "ok ": "ng "); uart_print_i8(rv); uart_print_endl();
}

void dcmd_start(char* args, void* param)
{
    (void)args;
    (void)param;
    int8_t rv = start();
    uart_print((rv == 0) ? "ok ": "ng "); uart_print_i8(rv); uart_print_endl();
}

void dcmd_status(char* args, void* param)
{
    (void)param;
    uint8_t force;
    uint8_t i;

    i = stou8(&force, args);
    if(i > 0 && force && chgv_status_cnt == 0)
    {
        write_status();
    }

    if(chgv_status_cnt > 0)
    {
        uart_print("ok ");
        chgv_print_status(&chgv_status[chgv_status_top], 1);
        if(++chgv_status_top >= CHGV_STAUS_NUM) chgv_status_top = 0;
        chgv_status_cnt--;
    }
    else
    {
        uart_print("ok"); uart_print_endl();
    }
}

void dcmd_test(char* args, void* param)
{
    (void)param;
    uint8_t i, test_no;
    uint8_t err = 0;
    int8_t rv;
    i = stou8(&test_no, args);
    if(i == 0)
    {
        err = 1;
        goto error_exit;
    }
    rv = exec_test(test_no);
    uart_print((rv == 0) ? "ok ": "ng ");
    uart_print_u8(test_no); uart_print(" "); uart_print_i8(rv);
    goto end;

error_exit:
    uart_print("ng "); uart_print_u8(err);

end:
    uart_print_endl();
}

void dcmd_execute(void)
{
    dcmd_buf_len += uart_recv((uint8_t*)dcmd_buf + dcmd_buf_len, DCMD_BUF_SIZE - dcmd_buf_len);
    if(dcmd_buf_len > 0)
    {
        uint8_t i, j, k;

        for(i = 0; i < dcmd_buf_len; i++)
        {
            if(dcmd_buf[i] == '\r')
            {
                dcmd_buf[i] = 0;
                break;
            }
        }
        if(i < dcmd_buf_len)
        {
            for(j = 0; j < CMD_COUNT; j++)
            {
                const dbg_cmd_t* cmd = &dcmd_list[j];
                if((k = strcontain(dcmd_buf, cmd->cmd)) > 0)
                {
                    dcmd_list[j].exec(dcmd_buf + k, cmd->param);
                    break;
                }
            }
            if(j >= CMD_COUNT)
            {
                uart_print("ng unknown, \""); uart_print(dcmd_buf);
                uart_print("\""); uart_print_endl();
            }

            // Erase the read message
            i++;
            for(j = 0; i < dcmd_buf_len; j++, i++)
            {
                dcmd_buf[j] = dcmd_buf[i];
            }
            dcmd_buf_len = j;
        }
    }
}
