// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>
#include "chgv.h"
#include "nvm.h"
#include "uart.h"
#include "utility.h"

chgv_config_t chgv_config;
chgv_status_t chgv_status[CHGV_STAUS_NUM];
uint8_t chgv_status_cnt = 0;
uint8_t chgv_status_top = 0;
char chgv_status_str[CHGV_STATUS_STR_LEN];

void chgv_cal_tgt_vol(void)
{
    // If bat_cap is too small, the voltage detection of q-charge doesn't work.
    // 700mA means Eneloop Lite scale factor should be 2.
    uint8_t scale_factor = (BAT_CAP >= 700) ? 1: 2;
    uint16_t scaled_bat_cap = BAT_CAP * scale_factor;

    // 0.5C (bat_cap[mA] * 0.5 * 0.1ohm * 16 / 1mV = bat_cap * 0.8)
    Q_CHG_TGT_VOL = scaled_bat_cap * 4 / 5;
    // 0.1C (bat_cap[mA] * 0.1 * 0.5ohm * 16 / 1mV = bat_cap * 0.8)
    N_CHG_TGT_VOL = scaled_bat_cap * 4 / 5;

    // DAC resolution = 1.024V / 256 = 4mV
    // 0.5C (bat_cap[mA] * 0.5 * 0.5ohm / 4mV = bat_cap / 16)
    DSC_TGT_STEP = (uint8_t)(scaled_bat_cap / 16);
    // 0.2C (bat_cap[mA] * 0.2 * 0.5ohm / 4mV = bat_cap / 40)
    DSC_END_TGT_STEP = (uint8_t)(scaled_bat_cap / 40);

     // [0.1sec], 16.33hour (Golden Power Spec)
    N_CHG_TIME = 587880 / scale_factor;
     // [0.1sec], 2.45 * 1.2hour (Golden Power Spec * margin)
    Q_CHG_TIME = 105840 / scale_factor;
}

void chgv_init_config(void)
{
    OPA_OFFSET       =      0; // use factory value
    Q_CHG_MIN_VOL    =    900; // [mV]
    DSC_ONLY         =      0;
    N_CHG_FIN_VOL    =   1100; //1200; // [mV]
    BAT_MIN_VOL      =    500; // [mV]
    MIN_CHG_STA_VOL  =   1000; // [mV]
    BAT_MAX_VOL      =   1500; // [mV]
    BAT_CAP          =   1900; // [mA], max 6553mA
    chgv_cal_tgt_vol();
    //Q_CHG_TGT_VOL
    //N_CHG_TGT_VOL
    //DSC_TGT_STEP
    //DSC_END_TGT_STEP
    DSC_END_VOL      =   1000; // [mV]
    Q_CHG_DET_VOL    =      5; // [mV]
    N_CHG_TIME       = 587880; // [0.1sec], 16.33hour
    Q_CHG_TIME       = 105840; // [0.1sec], 2.45 * 1.2hour
    Q_CHG_NO_CHK_DUR =  12000; // [0.1sec], 20min
    Q_CHG_NO_CHK_VOL =   1400; // [mV]
    IDLE_MES         =      0;
    CFG_SIGNATURE    = CHGV_CFG_SIGNATURE_CONST;
}

int8_t chgv_load_config(void)
{
    uint8_t calc_sum;
    uint8_t read_sum;

    nvm_read(&chgv_config, 0, CHGV_CFG_SIZE);
    nvm_read(&read_sum, CHGV_CFG_SIZE, 1);

    calc_sum = calc_checksum(&chgv_config, CHGV_CFG_SIZE);
    if(calc_sum != read_sum || CFG_SIGNATURE != CHGV_CFG_SIGNATURE_CONST)
    {
        uart_print("Config load error Set initial values.\r\n");
        chgv_init_config();
        return -1;
    }

    return 0;
}

int8_t chgv_verify_config(const uint8_t checksum)
{
    if(nvm_verify(&chgv_config, 0, CHGV_CFG_SIZE) != CHGV_CFG_SIZE)
    {
        return -1;
    }

    if(nvm_verify(&checksum, CHGV_CFG_SIZE, 1) != 1)
    {
        return -2;
    }

    return 0;
}

int8_t chgv_save_config(void)
{
    uint8_t sum = calc_checksum(&chgv_config, CHGV_CFG_SIZE);

    if(nvm_write(&chgv_config, 0, CHGV_CFG_SIZE) != CHGV_CFG_SIZE)
    {
        return -1;
    }

    if(nvm_write(&sum, CHGV_CFG_SIZE, 1) != 1)
    {
        return -2;
    }

    if(chgv_verify_config(sum) != 0)
    {
        return -3;
    }

    return 0;
}

void chgv_print_config(void)
{
    uart_print("n_chg_time=");
    uart_print_u24(chgv_config.n_chg_time);
    uart_print("\r\ndsc_only=");
    uart_print_u8(chgv_config.dsc_only);
    uart_print("\r\nq_chg_time=");
    uart_print_u32(chgv_config.q_chg_time);
    uart_print("\r\nopa_offset=");
    uart_print_u8(chgv_config.opa_offset);
    uart_print("\r\nbat_cap=");
    uart_print_u16(chgv_config.bat_cap);
    uart_print("\r\nbat_max_vol=");
    uart_print_u16(chgv_config.bat_max_vol);
    uart_print("\r\nbat_min_vol=");
    uart_print_u16(chgv_config.bat_min_vol);
    uart_print("\r\ndsc_end_vol=");
    uart_print_u16(chgv_config.dsc_end_vol);
    uart_print("\r\nmin_chg_sta_vol=");
    uart_print_u16(chgv_config.min_chg_sta_vol);
    uart_print("\r\nn_chg_fin_vol=");
    uart_print_u16(chgv_config.n_chg_fin_vol);
    uart_print("\r\nn_chg_tgt_vol=");
    uart_print_u16(chgv_config.n_chg_tgt_vol);
    uart_print("\r\nq_chg_det_vol=");
    uart_print_u16(chgv_config.q_chg_det_vol);
    uart_print("\r\nq_chg_min_vol=");
    uart_print_u16(chgv_config.q_chg_min_vol);
    uart_print("\r\nq_chg_no_chk_dur=");
    uart_print_u16(chgv_config.q_chg_no_chk_dur);
    uart_print("\r\nq_chg_no_chk_vol=");
    uart_print_u16(chgv_config.q_chg_no_chk_vol);
    uart_print("\r\nq_chg_tgt_vol=");
    uart_print_u16(chgv_config.q_chg_tgt_vol);
    uart_print("\r\ndsc_tgt_step=");
    uart_print_u8(chgv_config.dsc_tgt_step);
    uart_print("\r\ndsc_end_tgt_step=");
    uart_print_u8(chgv_config.dsc_end_tgt_step);
    uart_print("\r\nidle_mes=");
    uart_print_u8(chgv_config.idle_mes);
    uart_print("\r\nsignature=");
    uart_print_u8(CFG_SIGNATURE);
    uart_print("\r\n");
}

void chgv_print_status(const chgv_status_t* status, uint8_t value_only)
{
    uint8_t i;
    if(!value_only) uart_print("st ");
    i = u32tos2(chgv_status_str, status->tick, 9, 0);
    chgv_status_str[i++] = ' ';
    i += u8tos2(chgv_status_str + i, status->prev_state, 2, 0);
    chgv_status_str[i++] = ' ';
    i += u8tos2(chgv_status_str + i, status->state, 2, 0);
    chgv_status_str[i++] = ' ';
    i += u8tos2(chgv_status_str + i, status->err_code, 1, 0);
    chgv_status_str[i++] = ' ';
    i += u16tos2(chgv_status_str + i, status->open_bat_vol, 4, 0);
    chgv_status_str[i++] = ' ';
    i += u16tos2(chgv_status_str + i, status->chgdsc_bat_vol, 4, 0);
    chgv_status_str[i++] = ' ';
    i += u16tos2(chgv_status_str + i, status->chgdsc_current, 4, 0);
    chgv_status_str[i++] = ' ';
    i += u16tos2(chgv_status_str + i, status->prev_chgdsc_bat_vol, 4, 0);
    chgv_status_str[i++] = '\r';
    chgv_status_str[i++] = '\n';
    uart_send((uint8_t*)chgv_status_str, i, 1);
}
