// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "adc.h"
#include "pwm.h"
#include "chgv_if.h"

#define ADC_PCH_BAT_TOP   0x14 // RC4/ANC4
#define ADC_PCH_N_CHG_CUR 0x93 // RC3 (OPA1IN1+)
#define ADC_PCH_Q_CHG_CUR 0x8d // RB5 (OPA1IN0+)
#define ADC_PCH_DAC1      0x3d
#define ADC_PCH_DAC2      0x3a
#define ADC_PCH_AVSS      0x3b
#define ADC_PCH_OPA_OUT   0x82 // RA2 (OPA1IN2+), DAC1OUT2
#define ADC_PCH_FVR1      0x3e
#define ADC_PCH_FVR2      0x3f

__uint24 adc_acc[ADC_CH_CNT];
uint16_t adc_val[ADC_CH_CNT];
uint16_t adc_meas_wait_cnt;
uint16_t adc_acc_cnt;
uint16_t adc_tgt_vol;
uint16_t adc_tst_pwm_duty;
uint16_t adc_meas_wait_cnt;
int16_t adc_pwm_duty;
int16_t adc_e1;
int16_t adc_de1;
uint8_t adc_pch[ADC_CH_CNT];
uint8_t adc_ch_idx;
uint8_t adc_mode;
uint8_t adc_result;
uint8_t adc_err;
uint8_t adc_acc_bit; // Max 12 from 24bit adc_acc and 12bit ADC value

void adc_init(void)
{
    //-------------------------------------------------------------------------
    // ADC
    // Acquisition time = 10usec
    // Conversion time = 14 TAD + 2 TCY = 14 * 0.5usec + 2 * 15.625nsec = 7.03usec
    // ADC time per 1 ch = 17.03usec

    // Bit 4 - NREF: 0 VREF- is connected to AVSS
    // Bits 1:0 - PREF[1:0]: 11 VREF+ is connected to internal Fixed Voltage Reference (FVR) module
    ADREF = 0x03;

    // 10usec * 64MHz(Fosc) = 640
    ADACQ = 0;

    // TAD = 2 * (15 + 1) / 64M = 0.5usec
    ADCLK = 15;

    // Bits 4:0 - ACT[4:0]: 01000 CCP1_trigger
    ADACT = 0x08;

    // Bit 7 - CPON: 1 Charge Pump On when requested by the ADC
    ADCP = 0x80;

    // Bit 7 - ON: 1 ADC is enabled
    // Bit 2 - FM: 1 ADRES and ADPREV data are right justified
    ADCON0 = 0x84;

    while(!ADCPbits.CPRDY) {}

    //-------------------------------------------------------------------------
    // CCP1
    // Bit 7 - EN: 0 CCP is disabled
    // Bits 3:0 - MODE[3:0]: 1011 Compare Pulse output; clear TMR1(2)
    // 0000 1011
    CCP1CON = 0x0b;

    //-------------------------------------------------------------------------
    // TMR1
    // Bits 4:0 - CS[4:0]: 00001 FOSC/4
    T1CLK = 0x01;

    // Bits 5:4 - CKPS[1:0]: 11 1:8 Prescaler value
    // Bit 0 - ON-Timer On: 0 Disables Timer
    // 0011 0000
    T1CON = 0x30;
}

void adc_set_tgt_vol(uint16_t vol)
{
    adc_tgt_vol = vol;
}

void adc_stop(void)
{
    T1CONbits.ON = 0;
    PIR3bits.CCP1IF = 0;
    CCP1CONbits.EN = 0;
    INTCON0bits.GIE = 0;
    pwm_set_duty(0);
    PIE1bits.ADIE = 0;
    PIR1bits.ADIF = 0;
    INTCON0bits.GIE = 1;
    ADPCH = ADC_PCH_AVSS;
}

void adc_start(uint8_t mode)
{
    if(T1CONbits.ON)
    {
        adc_stop();
    }
    adc_mode = mode;
    switch(mode)
    {
    case ADC_MD_Q_CHG:
    case ADC_MD_Q_CHG_TST:
        // Duration 92msec, PWM enabled
        adc_pch[1] = ADC_PCH_Q_CHG_CUR;
        adc_pch[0] = ADC_PCH_BAT_TOP;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc_bit = 7;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 20 * 92 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 99; // 20KHz
        break;
    case ADC_MD_N_CHG:
    case ADC_MD_N_CHG_TST:
        // Duration 92msec, PWM enabled
        adc_pch[1] = ADC_PCH_N_CHG_CUR;
        adc_pch[0] = ADC_PCH_BAT_TOP;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc_bit = 7;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 20 * 92 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 99; // 20KHz
        break;
    case ADC_MD_OP_VOL:
        // Duration 8msec, PWM disabled
        adc_pch[1] = ADC_PCH_DAC1;
        adc_pch[0] = ADC_PCH_BAT_TOP;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc[1] = adc_acc[0] = 0;
        adc_acc_bit = 3;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 0;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 1999; // 1KHz
        break;
    case ADC_MD_OP_VOL2:
        // Duration 100msec, PWM disabled
        adc_pch[1] = ADC_PCH_DAC1;
        adc_pch[0] = ADC_PCH_BAT_TOP;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc[1] = adc_acc[0] = 0;
        adc_acc_bit = 3;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 100 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 1999; // 1KHz
        break;
    case ADC_MD_DSC:
        // Duration 92msec, PWM disabled
        adc_pch[1] = ADC_PCH_DAC1;
        adc_pch[0] = ADC_PCH_BAT_TOP;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc[1] = adc_acc[0] = 0;
        adc_acc_bit = 3;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 92 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 1999; // 1KHz
        break;
    case ADC_MD_OFF_CAL:
        // Duration 100msec, PWM disabled
        adc_pch[1] = ADC_PCH_FVR2;
        adc_pch[0] = ADC_PCH_N_CHG_CUR;
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc[1] = adc_acc[0] = 0;
        adc_acc_bit = 3;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 100 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        CCPR1 = 1999; // 1KHz
        pwm_set_duty(PWM_HIGH_OUT);
        break;
    }

    adc_result = ADC_RST_NONE;
    adc_err = ADC_ERR_NONE;
    adc_pwm_duty = adc_e1 = adc_de1 = 0;
    ADPCH = adc_pch[adc_ch_idx];
    // dummy run to enable CCP trigger
    while(ADCON0bits.GO) {}
    ADCON0bits.GO = 1;
    while(ADCON0bits.GO) {}
    PIR1bits.ADIF = 0;
    PIE1bits.ADIE = 1;
    PIR3bits.CCP1IF = 0;
    PIR3bits.TMR1IF = 0;
    TMR1L = TMR1H = 0;
    CCP1CONbits.EN = 1;
    T1CONbits.ON = 1;
}

void __interrupt(irq(AD)) adc_adcc_handler(void)
{
    PIR1bits.ADIF = 0;
    uint16_t adres = ADRES;
    ADPCH = ADC_PCH_AVSS; // do first to make the duration until the next PCH is set

    if(adc_ch_idx == (ADC_CH_CNT - 1))
    {
        if(adc_mode <= ADC_MD_PWM_ENABLED)
        {
            // PID Control for DCDC PWM
            int16_t adc_e0 = ((int16_t)adc_tgt_vol - (int16_t)adres);
            int16_t adc_de0 = adc_e0 - adc_e1;
            int16_t adc_dde0 = adc_de0 - adc_de1;
            adc_pwm_duty += (adc_de0 >> 1) + (adc_e0 >> 2) + (adc_dde0 << 1);
            if(adc_pwm_duty < 0)
            {
                adc_pwm_duty = 0;
            }
            if(adc_pwm_duty > PWM_MAX_DUTY) adc_pwm_duty = PWM_MAX_DUTY;
            if(adc_tst_pwm_duty > 0) adc_pwm_duty = (int16_t)adc_tst_pwm_duty;
            adc_e1 = adc_e0;
            adc_de1 = adc_de0;
            pwm_set_duty((uint16_t)adc_pwm_duty);
        }
    }

    if(adc_meas_wait_cnt == 0)
    {
        adc_acc[adc_ch_idx] += adres;
    }

    if(adc_ch_idx > 0)
    {
        // process next CH
        adc_ch_idx--;
        // 10usec * 64MHz(Fosc) = 640
        ADACQ = 639;
        ADPCH = adc_pch[adc_ch_idx];
        ADCON0bits.GO = 1;
        return;
    }

    if(adc_meas_wait_cnt > 0)
    {
        adc_meas_wait_cnt--;
    }
    else
    {
        adc_acc_cnt--;
    }

    if(adc_acc_cnt > 0)
    {
        // continue ADC, wait CCP trigger
        adc_ch_idx = ADC_CH_CNT - 1;
        ADACQ = 0;
        ADPCH = adc_pch[adc_ch_idx];
        return;
    }

    // ADC duration in a mode expired
    adc_val[0] = (uint16_t)(adc_acc[0] >> adc_acc_bit);
    adc_val[1] = (uint16_t)(adc_acc[1] >> adc_acc_bit);
    adc_result = ADC_RST_CMP;

    if(adc_mode == ADC_MD_Q_CHG_TST || adc_mode == ADC_MD_N_CHG_TST)
    {
        // continue ADC, wait CCP trigger
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 20 * 92 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        ADACQ = 0;
        ADPCH = adc_pch[adc_ch_idx];
    }
    else if(adc_mode == ADC_MD_OFF_CAL)
    {
        // continue ADC, wait CCP trigger
        adc_ch_idx = ADC_CH_CNT - 1;
        adc_acc_bit = 6;
        adc_acc_cnt = (1 << adc_acc_bit);
        adc_meas_wait_cnt = 100 - adc_acc_cnt;
        adc_acc[1] = adc_acc[0] = 0;
        ADACQ = 0;
        ADPCH = adc_pch[adc_ch_idx];
    }
    else
    {
        // Stop ADC trigger
        pwm_set_duty(0);
        T1CONbits.ON = 0;
        PIR3bits.CCP1IF = 0;
        CCP1CONbits.EN = 0;
        PIE1bits.ADIE = 0;
        PIR1bits.ADIF = 0;
        ADPCH = ADC_PCH_AVSS;
    }
}
