// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <stdlib.h>
#include "utility.h"

const char digit_c[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
uint8_t utos_buf[11];

uint8_t stoi16(int16_t* value, const char* str)
{
    int16_t v = 0;
    uint8_t i, j;
    int8_t sign = 0;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    if(str[i] == '-')
    {
        sign = 1;
        i++;
    }
    for(j = i; str[i]; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v *= 10;
            v += c - '0';
        }
        else
        {
            break;
        }
    }
    if(i > j)
    {
        *value = (sign) ? -v: v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t stou8(uint8_t* value, const char* str)
{
    uint8_t v = 0;
    uint8_t i, j;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    for(j = i; str[i]; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v *= 10;
            v += c - '0';
        }
        else
        {
            break;
        }
    }
    if(i > j)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t stou16(uint16_t* value, const char* str)
{
    uint16_t v = 0;
    uint8_t i, j;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    for(j = i; str[i]; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v *= 10;
            v += c - '0';
        }
        else
        {
            break;
        }
    }
    if(i > j)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t stou24(__uint24* value, const char* str)
{
    __uint24 v = 0;
    uint8_t i, j;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    for(j = i; str[i]; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v *= 10;
            v += c - '0';
        }
        else
        {
            break;
        }
    }
    if(i > j)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t stou32(uint32_t* value, const char* str)
{
    uint32_t v = 0;
    uint8_t i, j;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    for(j = i; str[i]; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v *= 10;
            v += c - '0';
        }
        else
        {
            break;
        }
    }
    if(i > j)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t htou32(uint32_t* value, const char* str)
{
    uint32_t v = 0;
    uint8_t i, j, k;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    j = i + 8;
    for(k = i; str[i] && i < j; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v <<= 4;
            v |= c - '0';
        }
        else if('A' <= c && c <= 'F')
        {
            v <<= 4;
            v |= c - 'A' + 10;
        }
        else if('a' <= c && c <= 'f')
        {
            v <<= 4;
            v |= c - 'a' + 10;
        }
        else
        {
            break;
        }
    }
    if(i > k)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t htou16(uint16_t* value, const char* str)
{
    uint16_t v = 0;
    uint8_t i, j, k;

    // skip space
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    j = i + 4;
    for(k = i; str[i] && i < j; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v <<= 4;
            v |= c - '0';
        }
        else if('A' <= c && c <= 'F')
        {
            v <<= 4;
            v |= c - 'A' + 10;
        }
        else if('a' <= c && c <= 'f')
        {
            v <<= 4;
            v |= c - 'a' + 10;
        }
        else
        {
            break;
        }
    }
    if(i > k)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t htou8(uint8_t* value, const char* str)
{
    uint8_t v = 0;
    uint8_t i, j, k;

    // skip spaces
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    j = i + 2;
    for(k = i; str[i] && i < j; i++)
    {
        char c = str[i];
        if('0' <= c && c <= '9')
        {
            v <<= 4;
            v |= c - '0';
        }
        else if('A' <= c && c <= 'F')
        {
            v <<= 4;
            v |= c - 'A' + 10;
        }
        else if('a' <= c && c <= 'f')
        {
            v <<= 4;
            v |= c - 'a' + 10;
        }
        else
        {
            break;
        }
    }
    if(i > k)
    {
        *value = v;
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t stobytes(uint8_t* buff, const uint8_t buff_len, const char* str, uint8_t* buff_count)
{
    uint8_t i, j, l;

    // skip spaces
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    *buff_count = 0;
    l = i;
    for(j = 0; str[i] && *buff_count < buff_len; i++, j++)
    {
        char c = str[i];
        uint8_t k = j >> 1;
        if('0' <= c && c <= '9')
        {
            buff[k] <<= 4;
            buff[k] |= c - '0';
        }
        else if('A' <= c && c <= 'F')
        {
            buff[k] <<= 4;
            buff[k] |= c - 'A' + 10;
        }
        else if('a' <= c && c <= 'f')
        {
            buff[k] <<= 4;
            buff[k] |= c - 'a' + 10;
        }
        else
        {
            break; // terminate by invalid characters
        }
        *buff_count = (j + 1) >> 1;
    }
    if(i > l)
    {
        return i;
    }
    else
    {
        return 0;
    }
}

uint16_t stobytes2(uint8_t* buff, const uint16_t buff_len, const char* str, uint16_t* buff_count)
{
    uint16_t i, j, l;

    // skip spaces
    for(i = 0; str[i] && str[i] == ' '; i++) {}

    *buff_count = 0;
    for(j = 0; str[i] && *buff_count < buff_len; i++, j++)
    {
        char c = str[i];
        uint16_t k = j >> 1;
        if('0' <= c && c <= '9')
        {
            buff[k] <<= 4;
            buff[k] |= c - '0';
        }
        else if('A' <= c && c <= 'F')
        {
            buff[k] <<= 4;
            buff[k] |= c - 'A' + 10;
        }
        else if('a' <= c && c <= 'f')
        {
            buff[k] <<= 4;
            buff[k] |= c - 'a' + 10;
        }
        else
        {
            break; // terminate by invalid characters
        }
        *buff_count = (j + 1) >> 1;
    }
    if(i > l)
    {
        return i;
    }
    else
    {
        return 0;
    }
}

uint8_t u32tohs(char* buff, const uint32_t v)
{
    return u32tohs2(buff, v, 1);
}

uint8_t u32tohs2(char* buff, const uint32_t v, const uint8_t null_term)
{
    int8_t shift;
    uint8_t i = 0;
    for(shift = 28; shift >= 0; shift -= 4)
    {
        buff[i++] = digit_c[(v >> shift) & 0xf];
    }
    if(null_term) buff[i] = 0;
    return i;
}

uint8_t u24tohs(char* buff, const __uint24 v)
{
    return u24tohs2(buff, v, 1);
}

uint8_t u24tohs2(char* buff, const __uint24 v, const uint8_t null_term)
{
    int8_t shift;
    uint8_t i = 0;
    for(shift = 20; shift >= 0; shift -= 4)
    {
        buff[i++] = digit_c[(v >> shift) & 0xf];
    }
    buff[i] = 0;
    return i;
}

uint8_t u16tohs(char* buff, const uint16_t v)
{
    return u16tohs2(buff, v, 1);
}

uint8_t u16tohs2(char* buff, const uint16_t v, const uint8_t null_term)
{
    int8_t shift;
    uint8_t i = 0;
    for(shift = 12; shift >= 0; shift -= 4)
    {
        buff[i++] = digit_c[(v >> shift) & 0xf];
    }
    if(null_term) buff[i] = 0;
    return i;
}

uint8_t u12tohs(char* buff, const uint16_t v)
{
    return u12tohs2(buff, v, 1);
}

uint8_t u12tohs2(char* buff, const uint16_t v, const uint8_t null_term)
{
    int8_t shift;
    uint8_t i = 0;
    for(shift = 8; shift >= 0; shift -= 4)
    {
        buff[i++] = digit_c[(v >> shift) & 0xf];
    }
    if(null_term) buff[i] = 0;
    return i;
}

uint8_t u8tohs(char* buff, const uint8_t v)
{
    return u8tohs2(buff, v, 1);
}

uint8_t u8tohs2(char* buff, const uint8_t v, const uint8_t null_term)
{
    int8_t shift;
    uint8_t i = 0;
    for(shift = 4; shift >= 0; shift -= 4)
    {
        buff[i++] = digit_c[(v >> shift) & 0xf];
    }
    if(null_term) buff[i] = 0;
    return i;
}

uint8_t bytestos(char* des, const uint8_t* src, const uint8_t src_len)
{
    uint8_t i, j;

    j = 0;
    for(i = 0; i < src_len; i++)
    {
        uint8_t v = src[i];
        des[j++] = digit_c[(v >> 4) & 0xf];
        des[j++] = digit_c[v & 0xf];
    }
    des[j] = 0;
    return j;
}

uint8_t u32tos2(char* buff, const uint32_t v, const uint8_t digit, const uint8_t null_term)
{
    uldiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = v;
    for(i = 0; x.quot > 0 && i < 10; i++)
    {
        x = uldiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    for(; j < k; j++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t u32tos(char* buff, const uint32_t v)
{
    return u32tos2(buff, v, 0, 1);
}

uint8_t i32tos2(char* buff, const int32_t v, const uint8_t digit, const uint8_t null_term)
{
    uldiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = (v < 0) ? (uint32_t)-v: (uint32_t)v;
    for(i = 0; x.quot > 0 && i < 10; i++)
    {
        x = uldiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    if(v < 0) buff[j++] = '-';
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t i32tos(char* buff, const int32_t v)
{
    return i32tos2(buff, v, 0, 1);
}

uint8_t u24tos2(char* buff, const __uint24 v, const uint8_t digit, const uint8_t null_term)
{
    uldiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = v;
    for(i = 0; x.quot > 0 && i < 8; i++)
    {
        x = uldiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t u24tos(char* buff, const __uint24 v)
{
    return u24tos2(buff, v, 0, 1);
}

uint8_t i24tos2(char* buff, const __int24 v, const uint8_t digit, const uint8_t null_term)
{
    uldiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = (v < 0) ? (uint32_t)-v: (uint32_t)v;
    for(i = 0; x.quot > 0 && i < 8; i++)
    {
        x = uldiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    if(v < 0) buff[j++] = '-';
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t i24tos(char* buff, const __int24 v)
{
    return  i24tos2(buff, v, 0, 1);
}

uint8_t u16tos2(char* buff, const uint16_t v, const uint8_t digit, const uint8_t null_term)
{
    udiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = v;
    for(i = 0; x.quot > 0 && i < 5; i++)
    {
        x = udiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t u16tos(char* buff, const uint16_t v)
{
    return u16tos2(buff, v, 0, 1);
}

uint8_t i16tos2(char* buff, const int16_t v, const uint8_t digit, const uint8_t null_term)
{
    udiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = (v < 0) ? (uint16_t)-v: (uint16_t)v;
    for(i = 0; x.quot > 0 && i < 5; i++)
    {
        x = udiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    if(v < 0) buff[j++] = '-';
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t i16tos(char* buff, const int16_t v)
{
    return i16tos2(buff, v, 0, 1);
}

uint8_t u8tos2(char* buff, const uint8_t v, const uint8_t digit, const uint8_t null_term)
{
    udiv_t x;
    uint8_t i, j;
    int8_t k, l;

    x.quot = v;
    for(i = 0; x.quot > 0 && i < 3; i++)
    {
        x = udiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t u8tos(char* buff, const uint8_t v)
{
    return u8tos2(buff, v, 0, 1);
}

uint8_t i8tos2(char* buff, const int8_t v, const uint8_t digit, const uint8_t null_term)
{
    udiv_t x;
    uint8_t i, j;
    int8_t k, l;

    if(v >= 0)
    {
        x.quot = (uint8_t)v;
    }
    else
    {
        x.quot = (uint8_t)(-v);
    }
    for(i = 0; x.quot > 0 && i < 3; i++)
    {
        x = udiv(x.quot, 10);
        utos_buf[i] = digit_c[(uint8_t)x.rem];
    }
    if(i == 0) utos_buf[i++] = '0';
    j = 0;
    if(v < 0) buff[j++] = '-';
    l = (int8_t)digit - (int8_t)i;
    for(k = 0; k < l; k++) buff[j++] = '0';
    while(i > 0)
    {
        buff[j++] = utos_buf[--i];
    }
    if(null_term) buff[j] = 0;

    return j;
}

uint8_t i8tos(char* buff, const int8_t v)
{
    return i8tos2(buff, v, 0, 1);
}

// Not terminated by null
uint8_t strcpy2(char* d, const char* s)
{
    uint8_t i;
    for(i = 0; s[i]; i++)
    {
        d[i] = s[i];
    }

    return i;
}

// If sub_str is matched with base_str from the first character,
// this function returns next character index to the matched characters.
// Otherwise, it returns 0.
uint8_t strcontain(const char* base_str, const char* sub_str)
{
    uint8_t i;
    for(i = 0; base_str[i] && sub_str[i]; i++)
    {
        if(base_str[i] != sub_str[i])
            break;
    }

    return (i > 0 && sub_str[i] == 0) ? i: 0;
}

uint8_t calc_checksum(const void* bytes, uint8_t len)
{
    uint8_t i;
    uint8_t calc_sum = 0;
    for(i = 0; i < len; i++)
    {
        calc_sum ^= ((uint8_t*)bytes)[i];
    }

    return calc_sum;
}
