// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "uart2.h"
#include "utility.h"

#define  UART2_RXBUF_SIZE 128
#define  UART2_TXBUF_SIZE 128

uint8_t uart2_rxbuf[UART2_RXBUF_SIZE];
uint8_t uart2_txbuf[UART2_TXBUF_SIZE];
volatile uint8_t uart2_rxbuf_rp = 0;
volatile uint8_t uart2_rxbuf_wp = 0;
volatile uint8_t uart2_txbuf_rp = 0;
volatile uint8_t uart2_txbuf_wp = 0;

void uart2_init(void)
{
    U2BRG = 7; // FCY = 64MHz, Baud Rate = 500kHz

    PIE8bits.U2RXIE = 1; // Set U1RX interrupt enable bit
    //PIE8bits.U2TXIE = 0; // Set U1TX interrupt enable bit

    // U1RX, U1TX default priority is high

    // Bit 5 - TXEN: 1 Transmit is enabled
    // Bit 4 - RXEN: 1 Receiver is enabled
    U2CON0 = 0x30;

    // Clear RX buffer
    U2FIFObits.RXBE = 1;

    // Bit 7 - ON: 1 Serial port enabled
    U2CON1 = 0x80;
}

void uart2_flush_rx_buf(void)
{
    uart2_rxbuf_rp = uart2_rxbuf_wp;
    U2FIFObits.RXBE = 1;
}

uint8_t uart2_recv(uint8_t* buff, const uint8_t size)
{
    uint8_t i = 0;
    while(uart2_rxbuf_rp != uart2_rxbuf_wp && i++ < size)
    {
        *buff++ = uart2_rxbuf[uart2_rxbuf_rp];
        if(++uart2_rxbuf_rp >= UART2_RXBUF_SIZE) uart2_rxbuf_rp = 0;
    }
    return i;
}

uint16_t uart2_send(const uint8_t* buff, const uint16_t size, const uint8_t sync)
{
    uint16_t i = 0;
    while(i < size)
    {
        uint8_t next_wp = uart2_txbuf_wp + 1;
        if(next_wp >= UART2_TXBUF_SIZE) next_wp = 0;
        if(next_wp != uart2_txbuf_rp)
        {
            uart2_txbuf[uart2_txbuf_wp] = buff[i++];
            uart2_txbuf_wp = next_wp;
        }
        else
        {
            if(sync)
            {
                volatile uint16_t j;
                if(uart2_txbuf_wp != uart2_txbuf_rp)
                {
                    // TX1IF is set when TX1REG gets to empty.
                    // But if TX1IF is cleared and TX1REG is still empty after that,
                    // TX1IF will not be set.
                    PIE8bits.U2TXIE = 1;
                }
                for(j = 0; j < 1000; j++) {} // wait
            }
            else
            {
                break; // If buffer is full, return.
            }
        }
    }
    if(uart2_txbuf_wp != uart2_txbuf_rp)
    {
        // TX1IF is set when TX1REG gets to empty.
        // But if TX1IF is cleared and TX1REG is still empty after that,
        // TX1IF will not be set.
        PIE8bits.U2TXIE = 1;
    }
    return i;
}

void __interrupt(irq(U2RX)) uart2_rx_int_handler(void)
{
    uint8_t v = U2RXB;
    uint8_t next_wp = uart2_rxbuf_wp + 1;
    if(next_wp >= UART2_RXBUF_SIZE) next_wp = 0;
    if(next_wp != uart2_rxbuf_rp)
    {
        uart2_rxbuf[uart2_rxbuf_wp] = v;
        uart2_rxbuf_wp = next_wp;
    }
    PIR8bits.U2RXIF = 0;
}

void __interrupt(irq(U2TX)) uart2_tx_int_handler(void)
{
    if(uart2_txbuf_wp != uart2_txbuf_rp)
    {
        U2TXB = uart2_txbuf[uart2_txbuf_rp];
        if(++uart2_txbuf_rp >= UART2_TXBUF_SIZE) uart2_txbuf_rp = 0;
    }
    else
    {
        PIE8bits.U2TXIE = 0;
    }
    PIR8bits.U2TXIF = 0;
}
