// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "uart1.h"
#include "utility.h"

#define  UART1_RXBUF_SIZE 128
#define  UART1_TXBUF_SIZE 128

uint8_t uart1_rxbuf[UART1_RXBUF_SIZE];
uint8_t uart1_txbuf[UART1_TXBUF_SIZE];
volatile uint8_t uart1_rxbuf_rp = 0;
volatile uint8_t uart1_rxbuf_wp = 0;
volatile uint8_t uart1_txbuf_rp = 0;
volatile uint8_t uart1_txbuf_wp = 0;

void uart1_init(void)
{
    U1BRG = 7; // FCY = 64MHz, Baud Rate = 500kHz

    PIE4bits.U1RXIE = 1; // Set U1RX interrupt enable bit
    //PIE4bits.U1TXIE = 0; // Set U1TX interrupt enable bit

    // U1RX, U1TX default priority is high

    // Bit 5 - TXEN: 1 Transmit is enabled
    // Bit 4 - RXEN: 1 Receiver is enabled
    U1CON0 = 0x30;

    // Clear RX buffer
    U1FIFObits.RXBE = 1;

    // Bit 7 - ON: 1 Serial port enabled
    U1CON1 = 0x80;

}

uint8_t uart1_recv(uint8_t* buff, const uint8_t size)
{
    uint8_t i = 0;
    while(uart1_rxbuf_rp != uart1_rxbuf_wp && i++ < size)
    {
        *buff++ = uart1_rxbuf[uart1_rxbuf_rp];
        if(++uart1_rxbuf_rp >= UART1_RXBUF_SIZE) uart1_rxbuf_rp = 0;
    }
    return i;
}

uint16_t uart1_send(const uint8_t* buff, const uint16_t size, const uint8_t sync)
{
    uint16_t i = 0;
    while(i < size)
    {
        uint8_t next_wp = uart1_txbuf_wp + 1;
        if(next_wp >= UART1_TXBUF_SIZE) next_wp = 0;
        if(next_wp != uart1_txbuf_rp)
        {
            uart1_txbuf[uart1_txbuf_wp] = buff[i++];
            uart1_txbuf_wp = next_wp;
        }
        else
        {
            if(sync)
            {
                volatile uint16_t j;
                if(uart1_txbuf_wp != uart1_txbuf_rp)
                {
                    // TX1IF is set when TX1REG gets to empty.
                    // But if TX1IF is cleared and TX1REG is still empty after that,
                    // TX1IF will not be set.
                    PIE4bits.U1TXIE = 1;
                }
                for(j = 0; j < 1000; j++) {} // wait
            }
            else
            {
                break; // If buffer is full, return.
            }
        }
    }
    if(uart1_txbuf_wp != uart1_txbuf_rp)
    {
        // TX1IF is set when TX1REG gets to empty.
        // But if TX1IF is cleared and TX1REG is still empty after that,
        // TX1IF will not be set.
        PIE4bits.U1TXIE = 1;
    }
    return i;
}

void __interrupt(irq(U1RX)) uart1_rx_int_handler(void)
{
    uint8_t v = U1RXB;
    uint8_t next_wp = uart1_rxbuf_wp + 1;
    if(next_wp >= UART1_RXBUF_SIZE) next_wp = 0;
    if(next_wp != uart1_rxbuf_rp)
    {
        uart1_rxbuf[uart1_rxbuf_wp] = v;
        uart1_rxbuf_wp = next_wp;
    }
    PIR4bits.U1RXIF = 0;
}

void __interrupt(irq(U1TX)) uart1_tx_int_handler(void)
{
    if(uart1_txbuf_wp != uart1_txbuf_rp)
    {
        U1TXB = uart1_txbuf[uart1_txbuf_rp];
        if(++uart1_txbuf_rp >= UART1_TXBUF_SIZE) uart1_txbuf_rp = 0;
    }
    else
    {
        PIE4bits.U1TXIE = 0;
    }
    PIR4bits.U1TXIF = 0;
}
