// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "switch.h"
#include "tmr.h"

#define SW_NUM           4
#define SW_ST_OFF        0
#define SW_ST_PRESS      1
#define SW_ST_LONG_PRESS 2

#define SW_START_STOP_BIT (PORTAbits.RA2)
#define SW_SET_BIT        (PORTAbits.RA3)
#define SW_DOWN_BIT       (PORTAbits.RA4)
#define SW_RETURN_BIT     (PORTAbits.RA5)

#define SW_BIT_PRESS   0
#define SW_BIT_RELEASE 1

#define LONG_PRESS_DURATION 2000 // ms
#define CHEKC_DURATION      10 // ms

uint8_t sw_state[SW_NUM];
uint16_t sw_press_start_tick[SW_NUM]; // effective less than 65536ms
uint16_t sw_update_tick = 0; // effective less than 65536ms
sw_request_t sw_req;

uint8_t inline sw_bit_state(uint8_t index)
{
    switch(index)
    {
        case 0:
            return SW_START_STOP_BIT;
        case 1:
            return SW_SET_BIT;
        case 2:
            return SW_DOWN_BIT;
        default: // 3
            return SW_RETURN_BIT;
    }
}

void sw_change_state(uint8_t index)
{
    if(sw_bit_state(index))
    {
        // High level: off
        sw_state[index] = SW_ST_OFF;
    }
    else
    {
        // Low level: on
        if(sw_state[index] == SW_ST_OFF)
        {
            sw_press_start_tick[index] = (uint16_t)tmr_get_tick();
            sw_state[index] = SW_ST_PRESS;
            sw_req.b = (uint8_t)(1 << (index << 1));
        }
        else if(sw_state[index] == SW_ST_PRESS)
        {
            if(((uint16_t)tmr_get_tick() - sw_press_start_tick[index]) >= (uint16_t)LONG_PRESS_DURATION)
            {
                // Note: SYSTM_TICK will warp around
                sw_state[index] = SW_ST_LONG_PRESS;
                sw_req.b = (uint8_t)(2 << (index << 1));
            }
        }
    }
}

void sw_update(void)
{
    if(((uint16_t)tmr_get_tick() - sw_update_tick) > CHEKC_DURATION)
    {
        uint8_t i;
        for(i = 0; i < SW_NUM; i++)
        {
            sw_change_state(i);
        }
        sw_update_tick = (uint16_t)tmr_get_tick();
    }
}
