#include "sdc.h"
#include "spi.h"
#include "uart1.h"
#include "uart_util.h"
#include "tmr.h"

#define SDC_TICK16 ((uint16_t)tmr_get_tick()) // msec resolution
#define SDC_TIMEOUT 1000 // 1sec
#define DEBUG_PRINT_ENABLE 0

#if DEBUG_PRINT_ENABLE
#define PRINT_DEBUG(...) uart_print(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_U32(...) uart_print_u32(__VA_ARGS__)
#define PRINT_DEBUG_HU32(...) uart_print_hu32(__VA_ARGS__)
#define PRINT_DEBUG_HU16(...) uart_print_hu16(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_HU8(...) uart_print_hu8(__VA_ARGS__)
#define PRINT_DEBUG_BYTES(...) uart_print_bytes(__VA_ARGS__)
#else
#define PRINT_DEBUG(...) {}
#define PRINT_DEBUG_U32(...) {}
#define PRINT_DEBUG_HU32(...) {}
#define PRINT_DEBUG_HU16(...) {}
#define PRINT_DEBUG_HU8(...) {}
#define PRINT_DEBUG_BYTES(...) {}
#endif

#define SDC_TX_BUF_SIZE 6
#define SDC_RX_BUF_SIZE 66
uint8_t sdc_tx_buf[SDC_TX_BUF_SIZE];
uint8_t sdc_rx_buf[SDC_RX_BUF_SIZE];

int8_t sdc_send_cmd(uint8_t res_len, uint8_t dummy_rd_len);
int8_t sdc_wait_busy();
int8_t sdc_rcv_data_pkt(uint8_t* buff, uint16_t block_size);

int8_t sdc_init(void)
{
    int8_t rv;

    // Send 74 byte clock
    spi_switch_baud_rate(0);
	spi_set_cs(1);
    spi_send_repeat(0xff, 74);

	spi_set_cs(0);
    sdc_tx_buf[0] = 0x40; // CMD0
    sdc_tx_buf[1] = 0x00;
    sdc_tx_buf[2] = 0x00;
    sdc_tx_buf[3] = 0x00;
    sdc_tx_buf[4] = 0x00;
    sdc_tx_buf[5] = 0x95;
    rv = sdc_send_cmd(1, 0);
    if(rv != 0)
    {
        rv = -1;
        goto err_exit;
    }
    uart_print_bytes(UART_DEV_DBG, sdc_rx_buf, 1); uart_print_endl(UART_DEV_DBG);
    if(sdc_rx_buf[0] != 0x01)
    {
        rv = -2;
        goto err_exit;
    }

    spi_switch_baud_rate(1);
    sdc_tx_buf[0] = 0x48; // CMD8
    sdc_tx_buf[1] = 0x00;
    sdc_tx_buf[2] = 0x00;
    sdc_tx_buf[3] = 0x01;
    sdc_tx_buf[4] = 0xaa;
    sdc_tx_buf[5] = 0x87; // CRC is always on
    rv = sdc_send_cmd(6, 0);
    if(rv != 0)
    {
        rv = -3;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x01)
    {
        rv = -4;
        goto err_exit;
    }

    sdc_tx_buf[0] = 0x7A; // CMD58
    sdc_tx_buf[1] = 0x00;
    sdc_tx_buf[2] = 0x00;
    sdc_tx_buf[3] = 0x00;
    sdc_tx_buf[4] = 0x00;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(5, 0);
    if(rv != 0)
    {
        rv = -5;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x01)
    {
        rv = -6;
        goto err_exit;
    }

    do
    {
        sdc_tx_buf[0] = 0x77; // CMD55
        sdc_tx_buf[1] = 0x00;
        sdc_tx_buf[2] = 0x00;
        sdc_tx_buf[3] = 0x00;
        sdc_tx_buf[4] = 0x00;
        sdc_tx_buf[5] = 0xFF;
        rv = sdc_send_cmd(1, 0);
        if(rv != 0)
        {
            rv = -7;
            goto err_exit;
        }
        if(sdc_rx_buf[0] != 0x01)
        {
            rv = -8;
            goto err_exit;
        }

        sdc_tx_buf[0] = 0x69; // ACMD41
        sdc_tx_buf[1] = 0x40; // SDHC or SDXC supported
        sdc_tx_buf[2] = 0x00;
        sdc_tx_buf[3] = 0x00;
        sdc_tx_buf[4] = 0x00;
        sdc_tx_buf[5] = 0xFF;
        rv = sdc_send_cmd(1, 0);
        if(rv != 0)
        {
            rv = -9;
            goto err_exit;
        }
        if(sdc_rx_buf[0] != 0x01 && sdc_rx_buf[0] != 0x00)
        {
            rv = -10;
            goto err_exit;
        }
    } while(sdc_rx_buf[0] != 0x00);

    sdc_tx_buf[0] = 0x77; // CMD55
    sdc_tx_buf[1] = 0x00;
    sdc_tx_buf[2] = 0x00;
    sdc_tx_buf[3] = 0x00;
    sdc_tx_buf[4] = 0x00;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(1, 0);
    if(rv != 0)
    {
        rv = -11;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x00)
    {
        rv = -12;
        goto err_exit;
    }

    sdc_tx_buf[0] = 0x4d; // ACMD13
    sdc_tx_buf[1] = 0x00;
    sdc_tx_buf[2] = 0x00;
    sdc_tx_buf[3] = 0x00;
    sdc_tx_buf[4] = 0x00;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(2, 0);
    if(rv != 0)
    {
        rv = -13;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x00 || sdc_rx_buf[1] != 0x00)
    {
        rv = -14;
        goto err_exit;
    }
    rv = sdc_rcv_data_pkt(sdc_rx_buf, 66);
    if(rv != 0)
    {
        rv = -15;
        goto err_exit;
    }
    PRINT_DEBUG("sdc_init\r\n");
    PRINT_DEBUG_BYTES(sdc_rx_buf, 66); PRINT_DEBUG("\r\n");

err_exit:
    spi_set_cs(1);
    return rv;
}

int8_t sdc_send_cmd(uint8_t res_len, uint8_t dummy_rd_len)
{
    uint8_t i;
    uint16_t start_tick;

    spi_transfer(sdc_tx_buf, sdc_rx_buf, 6);
    spi_send_repeat(0xff, dummy_rd_len);
    start_tick = (uint16_t)tmr_get_tick();
    do
    {
        sdc_rx_buf[0] = spi_transfer_byte(0xff);
        if(((uint16_t)tmr_get_tick() - start_tick) >= SDC_TIMEOUT)
        {
            return -1;
        }
    } while(sdc_rx_buf[0] == 0xff);
    for(i = 1; i < res_len; i++)
    {
        sdc_rx_buf[i] = spi_transfer_byte(0xff);
    }

    return 0;
}

int8_t sdc_wait_busy(void)
{
    uint16_t start_tick;
    uint16_t elapsed_tick;
    uint16_t cur_tick;
    start_tick = (uint16_t)tmr_get_tick();
    do
    {
        sdc_rx_buf[0] = spi_transfer_byte(0xff);
        cur_tick = (uint16_t)tmr_get_tick();
        elapsed_tick = cur_tick - start_tick;
        if(elapsed_tick >= SDC_TIMEOUT)
        {
            uart_print(UART_DEV_DBG, "TO start=0x"); uart_print_hu16(UART_DEV_DBG, start_tick);
            uart_print(UART_DEV_DBG, " cur=0x"); uart_print_hu16(UART_DEV_DBG, cur_tick);
            uart_print(UART_DEV_DBG, " elapse=0x"); uart_print_hu16(UART_DEV_DBG, elapsed_tick);
            uart_print_endl(UART_DEV_DBG);
            return -1;
        }
    } while(sdc_rx_buf[0] == 0);
    if(elapsed_tick > 100)
    {
        uart_print(UART_DEV_DBG, "elapsed_tick=0x");
        uart_print_hu16(UART_DEV_DBG, elapsed_tick); uart_print_endl(UART_DEV_DBG);
    }

    return 0;
}

int8_t sdc_rcv_data_pkt(uint8_t* buff, uint16_t block_size)
{
    do
    {
        sdc_rx_buf[0] = spi_transfer_byte(0xff);
        if((sdc_rx_buf[0] & 0xf0) == 0)
            return -1; // error token
    } while(sdc_rx_buf[0] != 0xfe);
    spi_receive(buff, block_size);
    // CRC16, read and discard
    spi_transfer_byte(0xff);
    spi_transfer_byte(0xff);

    return 0;
}

int8_t sdc_read_block(uint8_t* buff, uint32_t addr, uint16_t count)
{
    uint16_t i;
    int8_t rv = 0;

    PRINT_DEBUG("sdc_read_block addr=0x"); PRINT_DEBUG_HU32(addr);
    PRINT_DEBUG(" count=0x"); PRINT_DEBUG_HU32(count); PRINT_DEBUG("\r\n");
    spi_set_cs(0);
    sdc_tx_buf[0] = 0x52; // CMD18
    sdc_tx_buf[1] = (uint8_t)(addr >> 24);
    sdc_tx_buf[2] = (uint8_t)(addr >> 16);
    sdc_tx_buf[3] = (uint8_t)(addr >> 8);
    sdc_tx_buf[4] = (uint8_t)addr;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(1, 0);
    if(rv != 0)
    {
        rv = -1;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x00)
    {
        rv = -2;
        goto err_exit;
    }
    for(i = 0 ; i < count; i++)
    {
        rv = sdc_rcv_data_pkt(buff + (i << 9), 512);
        if(rv != 0)
        {
            rv = -2;
            goto err_exit;
        }
    }
    sdc_tx_buf[0] = 0x4c; // CMD12
    sdc_tx_buf[1] = 0;
    sdc_tx_buf[2] = 0;
    sdc_tx_buf[3] = 0;
    sdc_tx_buf[4] = 0;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(1, 1);
    if(rv != 0)
    {
        rv = -3;
        goto err_exit;
    }
    rv = sdc_wait_busy();
    if(rv != 0)
    {
        rv = -4;
        goto err_exit;
    }

err_exit:
    spi_set_cs(1);
    return rv;
}

int8_t sdc_write_block(uint8_t* buff, uint32_t addr, uint16_t count)
{
    uint16_t i;
    int8_t rv = 0;

    PRINT_DEBUG("sdc_write_block addr=0x"); PRINT_DEBUG_HU32(addr);
    PRINT_DEBUG(" count=0x"); PRINT_DEBUG_HU32(count); PRINT_DEBUG("\r\n");
    spi_set_cs(0);
    sdc_tx_buf[0] = 0x59; // CMD25
    sdc_tx_buf[1] = (uint8_t)(addr >> 24);
    sdc_tx_buf[2] = (uint8_t)(addr >> 16);
    sdc_tx_buf[3] = (uint8_t)(addr >> 8);
    sdc_tx_buf[4] = (uint8_t)addr;
    sdc_tx_buf[5] = 0xFF;
    rv = sdc_send_cmd(1, 0);
    if(rv != 0)
    {
        rv = -1;
        goto err_exit;
    }
    if(sdc_rx_buf[0] != 0x00)
    {
        rv = -2;
        goto err_exit;
    }
    spi_transfer_byte(0xff); // wait
    for(i = 0; i < count; i++)
    {
        spi_transfer_byte(0xfc); // start token
        spi_send(buff + (i << 9), 512);
        // Send dummy CRC
        spi_transfer_byte(0xff);
        spi_transfer_byte(0xff);
        // Get data response
        sdc_rx_buf[0] = spi_transfer_byte(0xff);
        if((sdc_rx_buf[0] & 0x0e) != 0x04)
        {
            // Data was not accepted
            rv = -3;
            goto err_exit;
        }
        rv = sdc_wait_busy();
        if(rv != 0)
        {
            rv = -4;
            goto err_exit;
        }
    }
    spi_transfer_byte(0xfd); // stop token
    spi_transfer_byte(0xff); // skip 1 byte
    rv = sdc_wait_busy();
    if(rv != 0)
    {
        rv = -5;
        goto err_exit;
    }

err_exit:
    spi_set_cs(1);
    return rv;
}

