
// PIC18F26Q84 Configuration Bit Settings

// 'C' source line config statements

// CONFIG1
#pragma config FEXTOSC = OFF    // External Oscillator Selection (Oscillator not enabled)
#pragma config RSTOSC = HFINTOSC_64MHZ// Reset Oscillator Selection (HFINTOSC with HFFRQ = 64 MHz and CDIV = 1:1)

// CONFIG2
#pragma config CLKOUTEN = OFF   // Clock out Enable bit (CLKOUT function is disabled)
#pragma config PR1WAY = ON      // PRLOCKED One-Way Set Enable bit (PRLOCKED bit can be cleared and set only once)
#pragma config CSWEN = OFF      // Clock Switch Enable bit (The NOSC and NDIV bits cannot be changed by user software)
#pragma config JTAGEN = OFF     // JTAG Enable bit (Disable JTAG Boundary Scan mode, JTAG pins revert to user functions)
#pragma config FCMEN = ON       // Fail-Safe Clock Monitor Enable bit (Fail-Safe Clock Monitor enabled)
#pragma config FCMENP = ON      // Fail-Safe Clock Monitor -Primary XTAL Enable bit (FSCM timer will set FSCMP bit and OSFIF interrupt on Primary XTAL failure)
#pragma config FCMENS = ON      // Fail-Safe Clock Monitor -Secondary XTAL Enable bit (FSCM timer will set FSCMS bit and OSFIF interrupt on Secondary XTAL failure)

// CONFIG3
#pragma config MCLRE = EXTMCLR  // MCLR Enable bit (If LVP = 0, MCLR pin is MCLR; If LVP = 1, RE3 pin function is MCLR )
#pragma config PWRTS = PWRT_64  // Power-up timer selection bits (PWRT set at 64ms)
#pragma config MVECEN = ON      // Multi-vector enable bit (Multi-vector enabled, Vector table used for interrupts)
#pragma config IVT1WAY = ON     // IVTLOCK bit One-way set enable bit (IVTLOCKED bit can be cleared and set only once)
#pragma config LPBOREN = OFF    // Low Power BOR Enable bit (Low-Power BOR disabled)
#pragma config BOREN = SBORDIS  // Brown-out Reset Enable bits (Brown-out Reset enabled , SBOREN bit is ignored)

// CONFIG4
#pragma config BORV = VBOR_1P9  // Brown-out Reset Voltage Selection bits (Brown-out Reset Voltage (VBOR) set to 1.9V)
#pragma config ZCD = OFF        // ZCD Disable bit (ZCD module is disabled. ZCD can be enabled by setting the ZCDSEN bit of ZCDCON)
#pragma config PPS1WAY = ON     // PPSLOCK bit One-Way Set Enable bit (PPSLOCKED bit can be cleared and set only once; PPS registers remain locked after one clear/set cycle)
#pragma config STVREN = ON      // Stack Full/Underflow Reset Enable bit (Stack full/underflow will cause Reset)
#pragma config LVP = OFF        // Low Voltage Programming Enable bit (HV on MCLR/VPP must be used for programming)
#pragma config XINST = OFF      // Extended Instruction Set Enable bit (Extended Instruction Set and Indexed Addressing Mode disabled)

// CONFIG5
#pragma config WDTCPS = WDTCPS_31// WDT Period selection bits (Divider ratio 1:65536; software control of WDTPS)
#pragma config WDTE = OFF       // WDT operating mode (WDT Disabled; SWDTEN is ignored)

// CONFIG6
#pragma config WDTCWS = WDTCWS_7// WDT Window Select bits (window always open (100%); software control; keyed access not required)
#pragma config WDTCCS = SC      // WDT input clock selector (Software Control)

// CONFIG7
#pragma config BBSIZE = BBSIZE_512// Boot Block Size selection bits (Boot Block size is 512 words)
#pragma config BBEN = OFF       // Boot Block enable bit (Boot block disabled)
#pragma config SAFEN = OFF      // Storage Area Flash enable bit (SAF disabled)
#pragma config DEBUG = OFF      // Background Debugger (Background Debugger disabled)

// CONFIG8
#pragma config WRTB = OFF       // Boot Block Write Protection bit (Boot Block not Write protected)
#pragma config WRTC = OFF       // Configuration Register Write Protection bit (Configuration registers not Write protected)
#pragma config WRTD = OFF       // Data EEPROM Write Protection bit (Data EEPROM not Write protected)
#pragma config WRTSAF = OFF     // SAF Write protection bit (SAF not Write Protected)
#pragma config WRTAPP = OFF     // Application Block write protection bit (Application Block not write protected)

// CONFIG9
#pragma config BOOTPINSEL = RC5 // CRC on boot output pin selection (CRC on boot output pin is RC5)
#pragma config BPEN = OFF       // CRC on boot output pin enable bit (CRC on boot output pin disabled)
#pragma config ODCON = OFF      // CRC on boot output pin open drain bit (Pin drives both high-going and low-going signals)

// CONFIG10
#pragma config CP = OFF         // PFM and Data EEPROM Code Protection bit (PFM and Data EEPROM code protection disabled)

// CONFIG11
#pragma config BOOTSCEN = OFF   // CRC on boot scan enable for boot area (CRC on boot will not include the boot area of program memory in its calculation)
#pragma config BOOTCOE = HALT   // CRC on boot Continue on Error for boot areas bit (CRC on boot will stop device if error is detected in boot areas)
#pragma config APPSCEN = OFF    // CRC on boot application code scan enable (CRC on boot will not include the application area of program memory in its calculation)
#pragma config SAFSCEN = OFF    // CRC on boot SAF area scan enable (CRC on boot will not include the SAF area of program memory in its calculation)
#pragma config DATASCEN = OFF   // CRC on boot Data EEPROM scan enable (CRC on boot will not include data EEPROM in its calculation)
#pragma config CFGSCEN = OFF    // CRC on boot Config fuses scan enable (CRC on boot will not include the configuration fuses in its calculation)
#pragma config COE = HALT       // CRC on boot Continue on Error for non-boot areas bit (CRC on boot will stop device if error is detected in non-boot areas)
#pragma config BOOTPOR = OFF    // Boot on CRC Enable bit (CRC on boot will not run)

// CONFIG12
#pragma config BCRCPOLT = hFF   // Boot Sector Polynomial for CRC on boot bits 31-24 (Bits 31:24 of BCRCPOL are 0xFF)

// CONFIG13
#pragma config BCRCPOLU = hFF   // Boot Sector Polynomial for CRC on boot bits 23-16 (Bits 23:16 of BCRCPOL are 0xFF)

// CONFIG14
#pragma config BCRCPOLH = hFF   // Boot Sector Polynomial for CRC on boot bits 15-8 (Bits 15:8 of BCRCPOL are 0xFF)

// CONFIG15
#pragma config BCRCPOLL = hFF   // Boot Sector Polynomial for CRC on boot bits 7-0 (Bits 7:0 of BCRCPOL are 0xFF)

// CONFIG16
#pragma config BCRCSEEDT = hFF  // Boot Sector Seed for CRC on boot bits 31-24 (Bits 31:24 of BCRCSEED are 0xFF)

// CONFIG17
#pragma config BCRCSEEDU = hFF  // Boot Sector Seed for CRC on boot bits 23-16 (Bits 23:16 of BCRCSEED are 0xFF)

// CONFIG18
#pragma config BCRCSEEDH = hFF  // Boot Sector Seed for CRC on boot bits 15-8 (Bits 15:8 of BCRCSEED are 0xFF)

// CONFIG19
#pragma config BCRCSEEDL = hFF  // Boot Sector Seed for CRC on boot bits 7-0 (Bits 7:0 of BCRCSEED are 0xFF)

// CONFIG20
#pragma config BCRCEREST = hFF  // Boot Sector Expected Result for CRC on boot bits 31-24 (Bits 31:24 of BCRCERES are 0xFF)

// CONFIG21
#pragma config BCRCERESU = hFF  // Boot Sector Expected Result for CRC on boot bits 23-16 (Bits 23:16 of BCRCERES are 0xFF)

// CONFIG22
#pragma config BCRCERESH = hFF  // Boot Sector Expected Result for CRC on boot bits 15-8 (Bits 15:8 of BCRCERES are 0xFF)

// CONFIG23
#pragma config BCRCERESL = hFF  // Boot Sector Expected Result for CRC on boot bits 7-0 (Bits 7:0 of BCRCERES are 0xFF)

// CONFIG24
#pragma config CRCPOLT = hFF    // Non-Boot Sector Polynomial for CRC on boot bits 31-24 (Bits 31:24 of CRCPOL are 0xFF)

// CONFIG25
#pragma config CRCPOLU = hFF    // Non-Boot Sector Polynomial for CRC on boot bits 23-16 (Bits 23:16 of CRCPOL are 0xFF)

// CONFIG26
#pragma config CRCPOLH = hFF    // Non-Boot Sector Polynomial for CRC on boot bits 15-8 (Bits 15:8 of CRCPOL are 0xFF)

// CONFIG27
#pragma config CRCPOLL = hFF    // Non-Boot Sector Polynomial for CRC on boot bits 7-0 (Bits 7:0 of CRCPOL are 0xFF)

// CONFIG28
#pragma config CRCSEEDT = hFF   // Non-Boot Sector Seed for CRC on boot bits 31-24 (Bits 31:24 of CRCSEED are 0xFF)

// CONFIG29
#pragma config CRCSEEDU = hFF   // Non-Boot Sector Seed for CRC on boot bits 23-16 (Bits 23:16 of CRCSEED are 0xFF)

// CONFIG30
#pragma config CRCSEEDH = hFF   // Non-Boot Sector Seed for CRC on boot bits 15-8 (Bits 15:8 of CRCSEED are 0xFF)

// CONFIG31
#pragma config CRCSEEDL = hFF   // Non-Boot Sector Seed for CRC on boot bits 7-0 (Bits 7:0 of CRCSEED are 0xFF)

// CONFIG32
#pragma config CRCEREST = hFF   // Non-Boot Sector Expected Result for CRC on boot bits 31-24 (Bits 31:24 of CRCERES are 0xFF)

// CONFIG33
#pragma config CRCERESU = hFF   // Non-Boot Sector Expected Result for CRC on boot bits 23-16 (Bits 23:16 of CRCERES are 0xFF)

// CONFIG34
#pragma config CRCERESH = hFF   // Non-Boot Sector Expected Result for CRC on boot bits 15-8 (Bits 15:8 of CRCERES are 0xFF)

// CONFIG35
#pragma config CRCERESL = hFF   // Non-Boot Sector Expected Result for CRC on boot bits 7-0 (Bits 7:0 of CRCERES are 0xFF)

// #pragma config statements should precede project file includes.
// Use project enums instead of #define for ON and OFF.

#include <xc.h>
#include <string.h>
#include "uart1.h"
#include "uart2.h"
#include "uart_util.h"
#include "utility.h"
#include "tmr.h"
#include "switch.h"
#include "lcd.h"
#include "chgv.h"
#include "common.h"
#include "spi.h"
#include "sdc.h"
#include "fsys.h"
#include "dcmd.h"
#include "led.h"
#include "i2ch.h"

#define SD_MAX_FILE_COUNT     10000
#define SD_UNMANAGED_CLUSTERS 4

#define SD_CARD_DET_BIT PORTAbits.RA7
#define SD_CARD_DET_PRESENT 1
#define SD_CARD_DET_NO_DISK 0

#define ST_READ_CHG_VAR     0
#define ST_CHARGE           1
#define ST_CHARGE_DETAIL     2
#define ST_CHK_BAT_CAP      3
#define ST_SET_SEL_SLOT     4
#define ST_SET_SEL_BAT_CAP  5
#define ST_FIN              6
#define ST_FIN_RST          7
#define ST_ERR              8

#define ERR_NONE          0
#define ERR_READ_CHG_VAR  1
#define ERR_SAVE_CHG_VAR  2
#define ERR_START         3
#define ERR_RESET         4
#define ERR_SYSTEM        5
#define ERR_NUM           6

#define FS_ST_UNINIT   0
#define FS_ST_RESERVED 1
#define FS_ST_READY    2
#define FS_ST_WRITE    3
#define FS_ST_ERR      4

#define FS_ERR_NONE   0
#define FS_ERR_SDINIT 1
#define FS_ERR_FSINIT 2
#define FS_ERR_OPEN   3
#define FS_ERR_WRITE  4
#define FS_ERR_CLOSE  5

#define BAT_CAP_STEP    50
#define BAT_CAP_MIN    500
#define BAT_CAP_MAX   2500

#define DISP_UPDATE_PER 1000

#define NOF_SLOTS 4
#if NOF_SLOTS > CHGV_MAX_NOF_SLOTS
    #error "NOF_SLOTS is up to CHGV_MAX_NOF_SLOTS."
#endif

#define SLOT_STR_LEN   3

void transit_state_read_chgv(void);
void transit_state_chk_bat_cap(void);
void transit_state_set_sel_slot(void);
void transit_state_sel_bat_cap(void);
void transit_state_charge(void);
void transit_state_charge_detail(void);
void transit_state_fin(void);
void transit_state_fin_rst(void);
void transit_state_err(void);

typedef struct _state_transition_entry
{
    uint8_t state;
    void (*func)(void);
} state_transition_entry_t;

const state_transition_entry_t state_trans_tbl[] =
{
    {ST_READ_CHG_VAR, transit_state_read_chgv},
    {ST_CHK_BAT_CAP, transit_state_chk_bat_cap},
    {ST_SET_SEL_SLOT, transit_state_set_sel_slot},
    {ST_SET_SEL_BAT_CAP, transit_state_sel_bat_cap},
    {ST_CHARGE, transit_state_charge},
    {ST_CHARGE_DETAIL, transit_state_charge_detail},
    {ST_FIN, transit_state_fin},
    {ST_FIN_RST, transit_state_fin_rst},
    {ST_ERR, transit_state_err},
};
#define STATE_TRANS_TBL_SIZE (sizeof(state_trans_tbl) / sizeof(state_trans_tbl[0]))

uint32_t start_tick[NOF_SLOTS];
uint16_t disp_tick;
uint16_t temp_bat_cap;
uint8_t state = ST_READ_CHG_VAR;
uint8_t slot_no;
uint8_t enable_print_st = 1;
uint8_t err_code = ERR_NONE;
uint8_t fs_st = FS_ST_UNINIT;
uint8_t fs_err = FS_ERR_NONE;
uint8_t update_chg_status = 0;
char slot_str[SLOT_STR_LEN] = {'s', 't', ' '};

const char* err_txt[ERR_NUM] =
{
    "NONE",
    "READ_CHG_VAR",
    "SAVE_CHG_VAR",
    "START",
    "RESET",
    "SYSTEM"
};

void set_lcd_sd_init(void);

void port_init(void)
{
    SPI1SDIPPS = 0x12; // RC2 (010 010)
    I2C1SCLPPS = 0x13; // RC3 (010 011)
    I2C1SDAPPS = 0x14; // RC4 (010 100)
    U1RXPPS = 0x16; // RC6 (010 110)
    U2RXPPS = 0x0b; // RB3 (001 011)
    RC0PPS = 0x31; // SPI1 SCK
    RC1PPS = 0x32; // SPI1 SDO
    RC3PPS = 0x37; // I2C1 SCL
    RC4PPS = 0x38; // I2C1 SDA
    RC7PPS = 0x20; // UART1 TX
    RB2PPS = 0x23; // UART2 TX

    // PPS lock
    INTCON0bits.GIE = 0; //Suspend interrupts
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 1; //Set PPSLOCKED bit
    INTCON0bits.GIE = 1; //Restore interrupts

    LATA = 0x40;   // 0100 0000: RA6 high
    WPUA = 0x3c;   // 0011 1100: RA5-2 pull-up
    TRISA = 0xbc;  // 1011 1100: RA6,1-0 output
    ANSELA = 0x00; // 0000 0000: RA7-0 digital IO

    LATB = 0x04;   // 0000 0100: RB2 high
    WPUB = 0x03;   // 0000 0011: RB1-0 pull-up
    TRISB = 0xcb;  // 1100 1011: RB5-4,2 output
    ANSELB = 0x00; // 0000 0000: RB7-0 digital IO

    LATC = 0x98;   // 1001 1000: RC7,4-3 high
    ODCONC = 0x18; // 0001 1000: RC4-3 open-drain
    WPUC = 0x38;   // 0011 1000: RC5-3 pull-up
    TRISC = 0x64;  // 0110 0100: RC7,RC4-3,RC1-0 output
    ANSELC = 0x00; // 0000 0000: RC7-0 digital IO
}

uint8_t check_charge_finished(void)
{
    uint8_t i;
    for(i = 0; i < NOF_SLOTS; i++)
    {
        uint8_t state = chgv_status[i].state;
        if(state >= CHGV_ST_NUM)
        {
            uart_print(UART_DEV_DBG, "Slot"); uart_print_u8(UART_DEV_DBG, i);
            uart_print(UART_DEV_DBG, " state error "); uart_print_u8(UART_DEV_DBG, state);
            uart_print_endl(UART_DEV_DBG);
            state = ST_ERR;
            err_code = ERR_SYSTEM;
            led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
        }
        else if(state != CHGV_ST_FIN &&
           state != CHGV_ST_FIN_MES &&
           state != CHGV_ST_ERR)
        {
            // charging
            break;
        }
    }
    return (i < NOF_SLOTS) ? 0: 1;
}

void reset_all_slots(void)
{
    uint8_t i;
    int8_t rv;
    for(i = 0; i < NOF_SLOTS; i++)
    {
        rv = chgv_reset(i);
        if(rv != 0)
        {
            uart_putc(UART_DEV_DBG, '#'); uart_print_u8(UART_DEV_DBG, i);
            uart_print(UART_DEV_DBG, " Reset error: "); uart_print_i8(UART_DEV_DBG, rv);
            uart_print_endl(UART_DEV_DBG);
            break;
        }
    }
    if(i < NOF_SLOTS)
    {
        state = ST_ERR;
        err_code = ERR_RESET;
        led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
    }
    else
    {
        state = ST_READ_CHG_VAR;
        err_code = ERR_NONE;
    }
    update_chg_status = 1; // LCD update in status update
}

void read_status_callback(uint8_t slot, char* st_str)
{
    int8_t rv;
    slot_str[2] = '0' + slot;
    uint8_t len = (uint8_t)strlen(st_str);

    if(fs_st == FS_ST_WRITE)
    {
        if(fs_write((uint8_t*)slot_str, SLOT_STR_LEN) != 0)
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_WRITE;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
        }
        if(fs_write((uint8_t*)st_str, len) != 0)
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_WRITE;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
        }
    }
    if(enable_print_st)
    {
        uart1_send((uint8_t*)slot_str, SLOT_STR_LEN, 1);
        uart1_send((uint8_t*)st_str, len, 1);
    }
    if((rv = chgv_set_status(slot, st_str)) < 0)
    {
        uart_print(UART_DEV_DBG, "chgv_set_status failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
    }
    if(chgv_status[slot].state == CHGV_ST_CHK)
    {
        start_tick[slot] = chgv_status[slot].tick;
    }
}

void get_charger_status(void)
{
    if(update_chg_status)
    {
        uint8_t i;

        //uart_print(UART_DEV_DBG, "get_charger_status"); uart_print_endl(UART_DEV_DBG);
        update_chg_status = 0;
        for(i = 0; i < NOF_SLOTS; i++)
        {
            chgv_read_status(i, read_status_callback);
        }
        lcd_update_flag = 1;
    }
}

uint32_t get_estimate_log_file_size(void)
{
    uint8_t i;
    __uint24 time = 0; // [0.1sec]

    for(i = 0; i < NOF_SLOTS; i++)
    {
        if(time < chgv_n_chg_time[i])
        {
            time = chgv_n_chg_time[i];
        }
    }

    // 160 Byte/sec, margin 1048576 Byte
    return (time / 10) * 160 + 1048576;
}

void check_sd_card(void)
{
    if(SD_CARD_DET_BIT == SD_CARD_DET_NO_DISK)
    {
        fs_deinit();
        fs_st = FS_ST_UNINIT;
        led_set_mode(LED_PORT_SD, LED_MD_OFF, 0);
        return;
    }

    // don't initialize during charge
    if(fs_st == FS_ST_UNINIT &&
        (state != ST_CHARGE && state != ST_CHARGE_DETAIL))
    {
        if(sdc_init() != 0)
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_SDINIT;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
            return;
        }
        // Formatting SDC takes 30-40sec. So we display the state here in the LCD.
        // Basically, LCD contents depends on 'state' and are written by set_lcd().
        set_lcd_sd_init();
        lcd_update_flag = 1;
        lcd_update();
        if(fs_init(SD_MAX_FILE_COUNT, SD_UNMANAGED_CLUSTERS, 0) != 0)
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_FSINIT;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
            lcd_update_flag = 1;
            return;
        }
        fs_st = FS_ST_READY;
        led_set_mode(LED_PORT_SD, LED_MD_OFF, 0);
        lcd_update_flag = 1;
    }
}

void open_file(void)
{
    if(fs_st == FS_ST_READY)
    {
        if(fs_open(get_estimate_log_file_size()) == 0)
        {
            fs_st = FS_ST_WRITE;
            led_set_mode(LED_PORT_SD, LED_MD_ON, 0);
        }
        else
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_OPEN;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
        }
    }
}

void close_file(void)
{
    if(fs_st == FS_ST_WRITE)
    {
        if(fs_close() == 0)
        {
            fs_st = FS_ST_READY;
            led_set_mode(LED_PORT_SD, LED_MD_OFF, 0);
        }
        else
        {
            fs_st = FS_ST_ERR;
            fs_err = FS_ERR_CLOSE;
            led_set_mode(LED_PORT_SD, LED_MD_N_BLINK, fs_err);
        }
    }
}

void transit_state_read_chgv(void)
{
    uint8_t i;
    for(i = 0; i < NOF_SLOTS; i++)
    {
        if(chgv_read_config(i) != 0)
        {
            state = ST_ERR;
            err_code = ERR_READ_CHG_VAR;
            lcd_update_flag = 1;
            led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
            break;
        }
    }
    if(i >= NOF_SLOTS)
    {
        slot_no = 0;
        state = ST_CHK_BAT_CAP;
        lcd_update_flag = 1;
    }
}

void transit_state_chk_bat_cap(void)
{
    uint8_t i;
    int8_t rv;

    if(sw_req.bits.startstop_press)
    {
        open_file();
        for(i = 0; i < NOF_SLOTS; i++)
        {
            rv = chgv_start(i);
            if(rv != 0)
            {
                uart_putc(UART_DEV_DBG, '#'); uart_print_u8(UART_DEV_DBG, i);
                uart_print(UART_DEV_DBG, " Start error: "); uart_print_i8(UART_DEV_DBG, rv);
                uart_print_endl(UART_DEV_DBG);
                break;
            }
        }
        if(i < NOF_SLOTS)
        {
            state = ST_ERR;
            err_code = ERR_START;
            led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
        }
        else
        {
            slot_no = 0;
            disp_tick = (uint16_t)tmr_get_tick();
            state = ST_CHARGE;
            led_set_mode(LED_PORT_STATE, LED_MD_ON, 0);
            update_chg_status = 1; // LCD update in status update
        }
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.down_press)
    {
        slot_no++;
        if(slot_no >= NOF_SLOTS) slot_no = 0;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.set_long_press)
    {
        state = ST_SET_SEL_SLOT;
        slot_no = NOF_SLOTS;
        lcd_update_flag = 1;
    }
}

void transit_state_set_sel_slot(void)
{
    uint8_t i;

    if(sw_req.bits.down_press)
    {
        slot_no++;
        if(slot_no >= (NOF_SLOTS + 1)) slot_no = 0;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.set_press)
    {
        state = ST_SET_SEL_BAT_CAP;
        // If slot_no == NOF_SLOTS (ALL slots), use slot 0 data.
        i = slot_no;
        if(i >= NOF_SLOTS) i = 0;
        temp_bat_cap = (chgv_bat_cap[i] / BAT_CAP_STEP) * BAT_CAP_STEP;
        if(temp_bat_cap < BAT_CAP_MIN) temp_bat_cap = BAT_CAP_MIN;
        if(temp_bat_cap > BAT_CAP_MAX) temp_bat_cap = BAT_CAP_MAX;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.return_press)
    {
        slot_no = 0;
        state = ST_CHK_BAT_CAP;
        lcd_update_flag = 1;
    }
}

void transit_state_sel_bat_cap(void)
{
    uint8_t i;
    int8_t rv;

    if(sw_req.bits.down_press)
    {
        temp_bat_cap -= BAT_CAP_STEP;
        if(temp_bat_cap < BAT_CAP_MIN) temp_bat_cap = BAT_CAP_MAX;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.set_press)
    {
        if(slot_no == NOF_SLOTS)
        {
            for(i = 0; i < NOF_SLOTS; i++)
            {
                chgv_bat_cap[i] = temp_bat_cap;
                rv = chgv_write_config(i);
                if(rv != 0)
                {
                    uart_putc(UART_DEV_DBG, '#'); uart_print_u8(UART_DEV_DBG, i);
                    uart_print(UART_DEV_DBG, "Write chg var error: ");
                    uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
                    break;
                }
                rv = chgv_save_config(i);
                if(rv != 0)
                {
                    uart_putc(UART_DEV_DBG, '#'); uart_print_u8(UART_DEV_DBG, i);
                    uart_print(UART_DEV_DBG, " Save NVM error: "); uart_print_i8(UART_DEV_DBG, rv);
                    uart_print_endl(UART_DEV_DBG);
                    break;
                }
            }
            if(i < NOF_SLOTS)
            {
                state = ST_ERR;
                err_code = ERR_SAVE_CHG_VAR;
                led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
            }
            else
            {
                slot_no = 0;
                state = ST_CHK_BAT_CAP;
            }
        }
        else
        {
            chgv_bat_cap[slot_no] = temp_bat_cap;
            rv = chgv_write_config(slot_no);
            if(rv != 0)
            {
                uart_print(UART_DEV_DBG, "Write chg var error "); uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
                state = ST_ERR;
                err_code = ERR_SAVE_CHG_VAR;
                led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
            }
            else
            {
                rv = chgv_save_config(slot_no);
                if(rv != 0)
                {
                    uart_putc(UART_DEV_DBG, '#'); uart_print_u8(UART_DEV_DBG, i);
                    uart_print(UART_DEV_DBG, " Save NVM error: "); uart_print_i8(UART_DEV_DBG, rv);
                    uart_print_endl(UART_DEV_DBG);
                    state = ST_ERR;
                    err_code = ERR_SAVE_CHG_VAR;
                    led_set_mode(LED_PORT_STATE, LED_MD_N_BLINK, err_code);
                }
                else
                {
                    state = ST_SET_SEL_SLOT;
                }
            }
        }
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.return_press)
    {
        state = ST_SET_SEL_SLOT;
        lcd_update_flag = 1;
    }
}

void transit_state_charge(void)
{
    if(check_charge_finished())
    {
        state = ST_FIN;
        lcd_update_flag = 1;
        led_set_mode(LED_PORT_STATE, LED_MD_OFF, 0);
    }
    if(sw_req.bits.down_press)
    {
        state = ST_CHARGE_DETAIL;
        slot_no = 0;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.startstop_long_press)
    {
        close_file();
        reset_all_slots();
    }
    else if(((uint16_t)tmr_get_tick() - disp_tick) > DISP_UPDATE_PER)
    {
        disp_tick = (uint16_t)tmr_get_tick();
        update_chg_status = 1; // LCD update in status update

        slot_no += 2;
        if(slot_no >= NOF_SLOTS) slot_no = 0;
    }
}

void transit_state_charge_detail(void)
{
    if(sw_req.bits.down_press)
    {
        slot_no++;
        if(slot_no >= NOF_SLOTS) slot_no = 0;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.return_press)
    {
        slot_no = 0;
        state = ST_CHARGE;
    }
    else if(sw_req.bits.startstop_long_press)
    {
        close_file();
        reset_all_slots();
    }
    else if(((uint16_t)tmr_get_tick() - disp_tick) > DISP_UPDATE_PER)
    {
        disp_tick = (uint16_t)tmr_get_tick();
        update_chg_status = 1; // LCD update in status update
    }
}

void transit_state_fin(void)
{
    close_file();
    if(sw_req.bits.down_press)
    {
        state = ST_FIN_RST;
        slot_no = 0;
        lcd_update_flag = 1;
    }
    else if(sw_req.bits.return_long_press)
    {
        close_file();
        reset_all_slots();
    }
}

void transit_state_fin_rst(void)
{
   if(sw_req.bits.down_press)
   {
       slot_no++;
       if(slot_no >= NOF_SLOTS) slot_no = 0;
       lcd_update_flag = 1;
   }
   else if(sw_req.bits.return_press)
   {
       slot_no = 0;
       state = ST_FIN;
       lcd_update_flag = 1;
   }
}

void transit_state_err(void)
{
    if(sw_req.bits.return_long_press)
    {
        reset_all_slots();
    }
}

void transit_state(void)
{
    uint8_t i;
    for(i = 0; i < STATE_TRANS_TBL_SIZE; i++)
    {
        if(state_trans_tbl[i].state == state)
        {
            state_trans_tbl[i].func();
            break;
        }
    }
    sw_req.b = 0; // clear switch request
}

const char* inline get_err_text()
{
    return (err_code < ERR_NUM) ? err_txt[err_code]: "UNKNOWN";
}

void set_lcd_sd_init(void)
{
    strcpy2(lcd_data_buf[0], "Initializing SDC");
    strcpy2(lcd_data_buf[1], "                ");
}

void set_lcd_status(char* buff, uint8_t slot)
{
    uint8_t state = chgv_status[slot].state;
    char varstr[5];

    buff[0] = '#';
    buff[1] = '0' + slot;
    buff[3] = chgv_st_char[chgv_status[slot].state];

    if(state == CHGV_ST_ERR)
    {
        strcpy2(buff + 5, chgv_err_text(chgv_status[slot].err_code));
    }
    else
    {
        u16tos2(varstr, chgv_status[slot].open_bat_vol, 4, 0);
        buff[5] = varstr[0];
        buff[6] = '.';
        buff[7] = varstr[1];
        buff[8] = varstr[2];
        buff[9] = 'V';
        u16tos2(varstr, chgv_status[slot].chgdsc_current, 4, 0);
        buff[11] = varstr[0];
        buff[12] = '.';
        buff[13] = varstr[1];
        buff[14] = varstr[2];
        buff[15] = 'A';
    }
}

void set_lcd_status2(char* buff, uint8_t slot)
{
    chgv_status_t* status = &chgv_status[slot];
    uint16_t duration = (uint16_t)((status->tick - start_tick[slot]) / 60000); // min    

    if(state != CHGV_ST_ERR)
    {
        int16_t int_res = (int16_t)status->chgdsc_bat_vol - (int16_t)status->open_bat_vol;
        if(int_res < 0) int_res = -int_res;
        int_res = (int16_t)(((int32_t)int_res * 1000) / status->chgdsc_current);
    
        u16tos2(buff, (uint16_t)int_res, 3, 0);
        buff[3] = 'm';
        buff[4] = 0x1e; // ?
    }
        
    u16tos2(buff + 6, duration, 3, 0);
    buff[9] = 'm';
    buff[10] = 'i';
    buff[11] = 'n';
}

void set_lcd(void)
{
    uint8_t i;

    if(!lcd_update_flag)
        return;

    for(i = 0; i < 2; i++)
    {
        strcpy2(lcd_data_buf[i], "                ");
    }

    switch(state)
    {
    case ST_READ_CHG_VAR:
        strcpy2(lcd_data_buf[0], "Reading chg var");
        break;
    case ST_CHK_BAT_CAP:
        strcpy2(lcd_data_buf[0], "Check BAT_CAP");
        lcd_data_buf[1][0] = '#';
        lcd_data_buf[1][1] = '0' + slot_no;
        i = 3;
        i += u16tos(&lcd_data_buf[1][i], chgv_bat_cap[slot_no]);
        i += strcpy2(&lcd_data_buf[1][i], "mA");
        break;
    case ST_SET_SEL_SLOT:
        strcpy2(lcd_data_buf[0], "Select slot");
        if(slot_no == NOF_SLOTS)
        {
            strcpy2(lcd_data_buf[1], "ALL");
        }
        else
        {
            lcd_data_buf[1][0] = '#';
            lcd_data_buf[1][1] = '0' + slot_no;
        }
        break;
    case ST_SET_SEL_BAT_CAP:
        strcpy2(lcd_data_buf[0], "Select BAT_CAP");
        i = u16tos(&lcd_data_buf[1][0], temp_bat_cap);
        i += strcpy2(&lcd_data_buf[1][i], "mA");
        break;
    case ST_CHARGE:
        set_lcd_status(lcd_data_buf[0], slot_no);
        set_lcd_status(lcd_data_buf[1], slot_no + 1);
        break;
    case ST_CHARGE_DETAIL:
    case ST_FIN_RST:
        set_lcd_status(lcd_data_buf[0], slot_no);
        set_lcd_status2(lcd_data_buf[1], slot_no);
        break;
    case ST_ERR:
        strcpy2(lcd_data_buf[0], "Error");
        strcpy2(lcd_data_buf[1], get_err_text());
        break;
    case ST_FIN:
        strcpy2(lcd_data_buf[0], "Finished");
        break;
    }
}

void main(void)
{
    port_init();
    uart1_init();
    uart2_init();
    i2ch_init();
    tmr_init();
    spi_init();

    INTCON0bits.GIE = 1;

    lcd_init();
    sdc_init();

    while(1)
    {
        dcmd_execute();
        sw_update();
        get_charger_status();
        check_sd_card();
        transit_state();
        set_lcd();
        lcd_update();
        led_update();
    }
}

