// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "i2ch.h"
#include "common.h"
#include "tmr.h"
#include "uart_util.h"

#define I2C_TIMEOUT 1000 //ms

//#define I2C_DEBUG

#define I2C_ERR_NONE           0
#define I2C_ERR_LEN           -1
#define I2C_ERR_PROC_TIMEOUT  -2
#define I2C_ERR_BUS_COLLISION -3
#define I2C_ERR_BUS_TIMEOOUT  -4
#define I2C_ERR_NACK          -5

uint8_t* i2c_data;
uint8_t i2c_data_pos;
uint8_t i2c_processing = 0;
int8_t i2c_err;
uint8_t i2c_cnt_flag;

void i2ch_init(void)
{
    // I2C Baud rate = 64MHz(FPRECLK) / (127 + 1) / 5 = 100kHz
    //I2C1BAUD = 127;
    // I2C Baud rate = 64MHz/4(FPRECLK) / (63 + 1) / 5 = 50kHz
    I2C1BAUD = 63;

    // Bit 7 - EN: 0 The I2C module is disabled
    // Bits 2:0 - MODE[2:0]: 100 I2C Host mode, 7-bit address
    I2C1CON0 = 0x04;

    // Bit 7 - ACKCNT: 1 I2CxCNT = 0 Not Acknowledge (NACK) copied to SDA output
    I2C1CON1 = 0x80;

    // Bits 3:2 - SDAHT[1:0]: 10 Minimum of 30 ns hold time on SDA after the falling SCL edge
    // follow MMC setting.
    I2C1CON2 = 0x08;

    // I2C_CLK: Fosc / 4
    //I2C1CLK = 0x01;

    // Bus Timeout is set to 64ms
    // Bits 3:0 - BTOC[3:0]: 0111 MFINTOSC (32 kHz)
    I2C1BTOC = 0x07;

    // Bit 7 - TOREC: 1 A BTO event will reset the I2C module and set BTOIF
    // Bit 6 - TOBY32: BTO time = TOTIME * TBTOCLK * 32 (DS descrption is typo-)
    // Bits 5:0 - TOTIME[5:0]: 2, BTO time = TOTIME * TBTOCLK * 32
    // 1100 0010
    I2C1BTO = 0xc2;

    I2C1CNTL = 0x00;
    I2C1CNTH = 0x00;

    I2C1ERR = 0x07; // Enable error interrupt
    PIE7bits.I2C1EIE = 1;
    PIE7bits.I2C1IE = 1;
    PIE7bits.I2C1TXIE = 1;
    PIE7bits.I2C1RXIE = 1;

    /* Silicon-Errata: Section: 1.3.2 */
    #warning "Refer to erratum DS80000870F: https://www.microchip.com/content/dam/mchp/documents/MCU08/ProductDocuments/Errata/PIC18F27-47-57Q43-Silicon-Errata-and-Datasheet-Clarifications-80000870J.pdf"
    I2C1PIEbits.SCIE = 0;
    I2C1PIEbits.PCIE = 0;
    I2C1CON0bits.EN = 1;
    __delay_us(1);
    __nop();
    __nop();
    __nop();
    __nop();
    __nop();
    __nop();
    I2C1PIRbits.SCIF = 0;
    I2C1PIRbits.PCIF = 0;
    I2C1PIEbits.PCIE = 1;

    I2C1PIEbits.CNTIE = 1;
}

void i2ch_wait_proccesing()
{
    uint16_t start_tick;

    start_tick = (uint16_t)tmr_get_tick();
    while(i2c_processing)
    {
#ifndef I2C_DEBUG
        if(((uint16_t)tmr_get_tick() - start_tick) >= I2C_TIMEOUT)
        {
            i2c_err = I2C_ERR_PROC_TIMEOUT;
            break;
        }
#endif
    }
}

int8_t i2ch_write(uint8_t* buf, const uint8_t dev_addr, const uint8_t len)
{

    if(len == 0) return I2C_ERR_LEN;

    I2C1STAT1 = 0x00; // Clear error flags
    I2C1PIR = 0x00;
    I2C1STAT1bits.CLRBF = 1; // Clear RX buf

    I2C1ADB1 = (uint8_t)(dev_addr << 1); // WRITE
    I2C1CNTL = len;
    I2C1CNTH = 0;
    i2c_data = buf;
    i2c_data_pos = 0;
    i2c_processing = 1;
    i2c_err = I2C_ERR_NONE;
    i2c_cnt_flag = 0;

    // Start I2C write
    I2C1CON0bits.S = 1;

    i2ch_wait_proccesing();

    return i2c_err;
}

int8_t i2ch_read(uint8_t* buf, const uint8_t dev_addr, const uint8_t len)
{
    if(len == 0) return I2C_ERR_LEN;

    I2C1STAT1 = 0x00; // Clear error flags
    I2C1PIR = 0x00;
    I2C1STAT1bits.CLRBF = 1; // Clear RX buf

    I2C1ADB1 = (uint8_t)(dev_addr << 1) | 1; // READ
    I2C1CNTL = len;
    I2C1CNTH = 0;
    i2c_data = buf;
    i2c_data_pos = 0;
    i2c_processing = 1;
    i2c_err = I2C_ERR_NONE;
    i2c_cnt_flag = 0;

    // Start I2C read
    I2C1CON0bits.S = 1;

    i2ch_wait_proccesing();

    return i2c_err;
}

void i2ch_process(void)
{
#ifdef I2C_DEBUG
    uart_print_hu8_async(PIR7);
    uart_print_hu8_async(I2C1PIR);
    uart_print_hu8_async(I2C1ERR);
    uart_print_hu8_async(I2C1STAT0);
    uart_print_hu8_async(I2C1STAT1);
    uart_print_hu8_async(I2C1CON1);
    uart_print_hu8_async(I2C1CNTL);
    uart_print_hu8_async(I2C1ADB1);
    uart_print_hu8_async((uint8_t)i2c_err);
    uart_print_hu8_async(i2c_processing);
    uart_print_endl_async();
#endif

    if(PIR7bits.I2C1TXIF)
    {
        I2C1TXB = i2c_data[i2c_data_pos++];
        PIR7bits.I2C1TXIF = 0;
    }
    else if(PIR7bits.I2C1RXIF)
    {
        i2c_data[i2c_data_pos++] = I2C1RXB;
        PIR7bits.I2C1RXIF = 0;
    }
    else if(I2C1PIRbits.CNTIF)
    {
        // Finished successfully
        I2C1PIRbits.CNTIF = 0;
        i2c_cnt_flag = 1;
        // Stop condition will be generated
    }
    else if(I2C1ERRbits.BCLIF)
    {
        i2c_err = I2C_ERR_BUS_COLLISION;
        I2C1ERR &= 0x0f; // clear error
        i2c_processing = 0;
    }
    else if(I2C1ERRbits.BTOIF)
    {
        i2c_err = I2C_ERR_BUS_TIMEOOUT;
        I2C1ERR &= 0x0f; // clear error
        // Stop condition will be generated
    }
    else if(I2C1ERRbits.NACKIF)
    {
        if(!i2c_cnt_flag)
        {
            i2c_err = I2C_ERR_NACK;
        }
        I2C1ERR &= 0x0f; // clear error
        // Stop condition will be generated
    }
    else if(I2C1PIRbits.PCIF)
    {
        // After data transfer was finished or the error occurred,
        // Stop condition is generated.
        I2C1PIRbits.PCIF = 0;
        i2c_processing = 0;
    }
    else
    {
        I2C1PIR = 0;
    }
}

void __interrupt(irq(I2C1E)) i2ch_error_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('E');
#endif
    i2ch_process();
}

void __interrupt(irq(I2C1RX)) i2ch_rx_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('R');
#endif
    i2ch_process();
}

void __interrupt(irq(I2C1TX)) i2ch_tx_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('T');
#endif
    i2ch_process();
}

void __interrupt(irq(I2C1)) i2ch_int_handler(void)
{
#ifdef I2C_DEBUG
    uart_putc_async('I');
#endif
    i2ch_process();
}
