#include "fsys.h"
#include "uart1.h"
#include "uart_util.h"
#include "sdc.h"
#include "utility.h"
#include "common.h"
#include <string.h>

#define RTC_AVAILABLE 0
#define ERROR_PRINT_ENABLE 1
#define INFO_PRINT_ENABLE 1
#define DEBUG_PRINT_ENABLE 0

#if ERROR_PRINT_ENABLE
#define PRINT_ERROR(...) uart_print(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_ERROR_ENDL() uart_print_endl(UART_DEV_DBG)
#define PRINT_ERROR_U32(...) uart_print_u32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_ERROR_HU32(...) uart_print_hu32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_ERROR_I8(...) uart_print_i8(UART_DEV_DBG, __VA_ARGS__)
#else
#define PRINT_ERROR(...) {}
#define PRINT_ERROR_ENDL() {}
#define PRINT_ERROR_U32(...) {}
#define PRINT_ERROR_HU32(...) {}
#define PRINT_ERROR_I8(...) {}
#endif

#if INFO_PRINT_ENABLE
#define PRINT_INFO(...) uart_print(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_INFO_ENDL() uart_print_endl(UART_DEV_DBG)
#define PRINT_INFO_U32(...) uart_print_u32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_INFO_HU32(...) uart_print_hu32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_INFO_HU16(...) uart_print_hu16(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_INFO_HU8(...) uart_print_hu8(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_INFO_I8(...) uart_print_i8(UART_DEV_DBG, __VA_ARGS__)
#else
#define PRINT_INFO(...) {}
#define PRINT_INFO_ENDL() {}
#define PRINT_INFO_U32(...) {}
#define PRINT_INFO_HU32(...) {}
#define PRINT_INFO_HU16(...) {}
#define PRINT_INFO_HU8(...) {}
#define PRINT_INFO_I8(...) {}
#endif

#if DEBUG_PRINT_ENABLE
#define PRINT_DEBUG(...) uart_print(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_ENDL() uart_print_endl(UART_DEV_DBG)
#define PRINT_DEBUG_U32(...) uart_print_u32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_HU32(...) uart_print_hu32(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_HU16(...) uart_print_hu16(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_HU8(...) uart_print_hu8(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_U16(...) uart_print_u16(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_U8(...) uart_print_u8(UART_DEV_DBG, __VA_ARGS__)
#else
#define PRINT_DEBUG(...) {}
#define PRINT_DEBUG_ENDL() {}
#define PRINT_DEBUG_U32(...) {}
#define PRINT_DEBUG_HU32(...) {}
#define PRINT_DEBUG_HU16(...) {}
#define PRINT_DEBUG_HU8(...) {}
#define PRINT_DEBUG_U16(...) {}
#define PRINT_DEBUG_U8(...) {}
#endif

#define SEC_SIZE           512
#define FAT_ENT_SIZE       4
#define DIR_ENT_SIZE       32
#define FAT_ENT_PER_SEC    (SEC_SIZE / FAT_ENT_SIZE)
#define DIR_ENT_PER_SEC    (SEC_SIZE / DIR_ENT_SIZE)
#define LOG_CFG_CLUSTERS   1 // assume the file is less than 1 cluster.

#define FAT0            0x0ffffff8 // LSB byte is Medeia Type (F8)
#define FAT_EOC         0x0fffffff
#define FAT_BAD_CLUSTER 0x0ffffff7

#define DIR_ATTR_READ_ONLY  0x01
#define DIR_ATTR_HIDDEN     0x02
#define DIR_ATTR_SYSTEM     0x04
#define DIR_ATTR_VOLUME_ID  0x08
#define DIR_ATTR_DIRECTORY  0x10
#define DIR_ATTR_ARCHIVE    0x20

#define LOGGER_VOL_LAB "logger00001"
#define LOG_CFG_SIG 0xaa551869

#pragma pack(push, 1)
typedef struct _dir_t
{
    char Name[11];
    uint8_t Attr;
    uint8_t NTRes;
    uint8_t CrtTimeTenth;
    uint16_t CrtTime;
    uint16_t CrtDate;
    uint16_t LstAccDate;
    uint16_t FstClusHI;
    uint16_t WrtTime;
    uint16_t WrtDate;
    uint16_t FstClusLO;
    uint32_t FileSize;
} dir_t;

typedef struct _log_cfg_t
{
    uint32_t signature;
    uint32_t max_log_files;
    uint32_t root_dir_clusters;
    uint32_t start_log_file_no;
    uint32_t current_log_files;
    uint32_t log_file_clusters;
    uint32_t fs_next_free_clus;
    uint32_t fs_free_clusters;
    uint8_t filename_digit;
} log_cfg_t;
#pragma pack(pop)

uint32_t fs_bpb_sec;
uint32_t fs_bpb_tot_sec32;
uint32_t fs_bpb_fat_sz32; // could be more than fs_valid_fat_entries.
uint32_t fs_bpb_root_clus;
log_cfg_t fs_log_cfg;
uint16_t fs_bpb_bytes_per_sec;
uint16_t fs_bpb_rsvd_sec_cnt;
uint16_t fs_bpb_fs_info;
uint16_t fs_bpb_bk_boot_sec;
uint8_t fs_bpb_sec_per_clus;
uint8_t fs_bpb_num_fats;
char fs_bpb_vol_lab[11 + 1];
char fs_bpb_fil_sys_type[8 + 1];
uint32_t fs_valid_fat_entries;
uint16_t fs_dir_ent_per_clus;
uint32_t fs_bytes_per_clus;
uint8_t fs_initialized;
uint8_t fs_file_opened;
uint8_t fs_buf[SEC_SIZE];
char fs_filename_buf[11 + 1];

uint32_t fs_log_write_sec;
uint32_t fs_log_file_size;
uint16_t fs_log_cached_size;

#if !(RTC_AVAILABLE)
uint16_t latest_date = 0x21; // 1980/1/1
uint16_t latest_time = 0;
#endif

inline uint32_t data_sec(void)
{
    return fs_bpb_sec + fs_bpb_rsvd_sec_cnt + fs_bpb_fat_sz32 * fs_bpb_num_fats;
}

inline uint32_t log_cfg_sec(void)
{
    return data_sec() + fs_log_cfg.root_dir_clusters * fs_bpb_sec_per_clus;
}

inline uint32_t fat_sec(void)
{
    return fs_bpb_sec + fs_bpb_rsvd_sec_cnt;
}

inline uint32_t log_file_sec(void)
{
    return data_sec() +
           (fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS) * fs_bpb_sec_per_clus;
}

inline uint32_t fs_sec(void)
{
    return fs_bpb_sec + fs_bpb_fs_info;
}

inline uint32_t clus_to_sec(uint32_t cluster)
{
    return data_sec() + (cluster - fs_bpb_root_clus) * fs_bpb_sec_per_clus;
}

inline uint32_t log_file_no_to_dir_ent_index(uint32_t file_no)
{
    return 2 + file_no; // Volume info + LOG.CFG
}

inline uint32_t dir_ent_index_to_sec(uint32_t dir_ent_index)
{
    return data_sec() + dir_ent_index / DIR_ENT_PER_SEC;
}

inline uint32_t log_file_clus_add(uint32_t a, uint32_t b)
{
    // log file clusters: from fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS,
    //                    to fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS + fs_log_cfg.log_file_clusters
    uint32_t c = a + b;
    uint32_t log_file_clus_end = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS + fs_log_cfg.log_file_clusters;
    if(c >= log_file_clus_end)
    {
        c = c - log_file_clus_end +
            fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS;
    }
    return c;
}

inline uint32_t fat_ent_to_sec(uint32_t fat_ent, uint8_t fat_no)
{
    return fs_bpb_sec + fs_bpb_rsvd_sec_cnt + fs_bpb_fat_sz32 * fat_no + fat_ent / FAT_ENT_PER_SEC;
}

#if RTC_AVAILABLE
void fs_set_lastest_datetime(const uint16_t date, const uint16_t time)
{
    (void)date;
    (void)time;
    // Nothing to do
}

void fs_get_cur_datetime(uint16_t* date, uint16_t* time, uint8_t* tenth)
{
    uint32_t rtc_date;
    uint32_t rtc_time;
    uint16_t rtc_msec;
    // Get rtc_date, rtc_time, rtc_msec
    if(date != NULL)
    {
        // RTC date: YYMMDD0W
        // FAT Date
        // Bits 0-4: Day of month, valid value range 1-31 inclusive.
        // Bits 5-8: Month of year, 1 = January, valid value range 1-12 inclusive.
        // Bits 9-15: Count of years from 1980, valid value range 0-127 inclusive (1980-2107).
        *date = (uint16_t)(((((rtc_date >> 28) & 0xf) * 10 + ((rtc_date >> 24) & 0xf) + 20) << 9) | // year
           ((((rtc_date >> 20) & 0xf) * 10 + ((date >> 16) & 0xf)) << 5) | // month
           ((((rtc_date >> 12) & 0xf) * 10 + ((date >> 8) & 0xf)))); // day
    }
    if(time != NULL)
    {
        // RTC time: HHmmss00
        // FAT time
        // Bits 0-4: 2-second count, valid value range 0-29 inclusive (0 - 58 seconds).
        // Bits 5-10: Minutes, valid value range 0-59 inclusive.
        // Bits 11-15: Hours, valid value range 0-23 inclusive.
        *time = (uint16_t)(((((rtc_time >> 28) & 0xf) * 10 + ((rtc_time >> 24) & 0xf)) << 11) | // hour
                   ((((rtc_time >> 20) & 0xf) * 10 + ((rtc_time >> 16) & 0xf)) << 5) | // minutes
                   ((((rtc_time >> 12) & 0xf) * 10 + ((rtc_time >> 8) & 0xf)) >> 2)); // seconds
    }
    if(tenth != NULL)
    {
        // time: RTC time, HHmmss00,
        // msec: RTC msec
        // FAT time tenth: 0 - 199 (hundreds of a second?)
        *tenth = (uint8_t)((((time >> 8) & 0x1) * 100) + (msec / 10));
    }
}
#else
void fs_increment_datetime(void)
{
  const uint8_t mon_table[2][12] =
  {
    //1   2   3   4   5   6   7   8   9  10  11  12
    {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
    {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }
  };

  // Bits 0-4: 2-second count, valid value range 0-29 inclusive (0 - 58 seconds).
  // Bits 5-10: Minutes, valid value range 0-59 inclusive.
  // Bits 11-15: Hours, valid value range 0-23 inclusive.
  uint8_t sec = (uint8_t)((latest_time & 0x001f) << 1);
  uint8_t min = (uint8_t)((latest_time >> 5) & 0x003f);
  uint8_t hour = (uint8_t)((latest_time >> 11) & 0x001f);

  // Bits 0-4: Day of month, valid value range 1-31 inclusive.
  // Bits 5-8: Month of year, 1 = January, valid value range 1-12 inclusive.
  // Bits 9-15: Count of years from 1980, valid value range 0-127 inclusive (1980-2107).
  uint8_t day = (uint8_t)(latest_date & 0x001f);
  uint8_t mon = (uint8_t)((latest_date >> 5) & 0x000f);
  uint8_t year = (uint8_t)(latest_date >> 9);

  uint8_t leap_year = (year & 0x3) ? 1 : 0;

  sec++;
  if (sec >= 30)
  {
    sec = 0;
    min++;
  }
  if (min >= 60)
  {
    min = 0;
    hour++;
  }
  if (hour >= 24)
  {
    hour = 0;
    day++;
  }
  if ((day + 1) >= mon_table[leap_year][mon - 1])
  {
    day = 1;
    mon++;
  }
  if (mon >= 13)
  {
    mon = 1;
    year++;
  }

  latest_time = (uint16_t)hour << 11 | (uint16_t)min << 5 | (sec >> 1);
  latest_date = (uint16_t)year << 9 | (uint16_t)mon << 5 | day;
}

void fs_set_lastest_datetime(const uint16_t date, const uint16_t time)
{
  latest_date = date;
  latest_time = time;
}

void fs_get_cur_datetime(uint16_t* date, uint16_t* time, uint8_t* tenth)
{
  fs_increment_datetime();
  if (date != NULL) *date = latest_date;
  if (time != NULL) *time = latest_time;
  if (tenth != NULL) *tenth = 0;
}
#endif

int8_t fs_get_mbr_bpb()
{
    int8_t rv = 0;
    PRINT_DEBUG("\r\nfs_get_mbr_bpb\r\n");

    // Read MBR
    rv = sdc_read_block(fs_buf, 0, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }
    if(fs_buf[511] != 0x55 && fs_buf[511] != 0xAA)
    {
        rv = -2;
        goto err_exit;
    }

    // Read first partition from MBR
    memcpy(&fs_bpb_sec, fs_buf + 0x01be + 0x08, 4);
    PRINT_DEBUG("fs_bpb_sec=0x"); PRINT_DEBUG_HU32(fs_bpb_sec); PRINT_DEBUG_ENDL();

    // Read BPB
    rv = sdc_read_block(fs_buf, fs_bpb_sec, 1);
    if(rv != 0)
    {
        rv = -3;
        goto err_exit;
    }
    if(fs_buf[511] != 0x55 && fs_buf[511] != 0xAA)
    {
        rv = -4;
        goto err_exit;
    }
    memcpy(&fs_bpb_bytes_per_sec, fs_buf + 11, 2);
    memcpy(&fs_bpb_sec_per_clus, fs_buf + 13, 1);
    memcpy(&fs_bpb_rsvd_sec_cnt, fs_buf + 14, 2);
    memcpy(&fs_bpb_num_fats, fs_buf + 16, 1);
    memcpy(&fs_bpb_tot_sec32, fs_buf + 32, 4);
    memcpy(&fs_bpb_fat_sz32, fs_buf + 36, 4);
    memcpy(&fs_bpb_root_clus, fs_buf + 44, 4);
    memcpy(&fs_bpb_fs_info, fs_buf + 48, 2);
    memcpy(&fs_bpb_bk_boot_sec, fs_buf + 50, 2);
    memcpy(fs_bpb_vol_lab, fs_buf + 71, 11); fs_bpb_vol_lab[11] = 0;
    memcpy(fs_bpb_fil_sys_type, fs_buf + 82, 8); fs_bpb_fil_sys_type[8] = 0;
    fs_valid_fat_entries = (fs_bpb_tot_sec32 - fs_bpb_rsvd_sec_cnt - fs_bpb_fat_sz32 * fs_bpb_num_fats) / fs_bpb_sec_per_clus + fs_bpb_root_clus;
    fs_dir_ent_per_clus = fs_bpb_sec_per_clus * DIR_ENT_PER_SEC;
    fs_bytes_per_clus = (uint32_t)fs_bpb_sec_per_clus * SEC_SIZE;

    PRINT_INFO("bytes_per_sec=0x"); PRINT_INFO_HU32(fs_bpb_bytes_per_sec); PRINT_INFO_ENDL();
    PRINT_INFO("sec_per_clus=0x"); PRINT_INFO_HU8(fs_bpb_sec_per_clus); PRINT_INFO_ENDL();
    PRINT_INFO("rsvd_sec_cnt=0x"); PRINT_INFO_HU16(fs_bpb_rsvd_sec_cnt); PRINT_INFO_ENDL();
    PRINT_INFO("num_fats=0x"); PRINT_INFO_HU8(fs_bpb_num_fats); PRINT_INFO_ENDL();
    PRINT_INFO("tot_sec32=0x"); PRINT_INFO_HU32(fs_bpb_tot_sec32); PRINT_INFO_ENDL();
    PRINT_INFO("fat_sz32=0x"); PRINT_INFO_HU32(fs_bpb_fat_sz32); PRINT_INFO_ENDL();
    PRINT_INFO("root_clus=0x"); PRINT_INFO_HU32(fs_bpb_root_clus); PRINT_INFO_ENDL();
    PRINT_INFO("fs_info=0x"); PRINT_INFO_HU16(fs_bpb_fs_info); PRINT_INFO_ENDL();
    PRINT_INFO("bk_boot_sec=0x"); PRINT_INFO_HU16(fs_bpb_bk_boot_sec); PRINT_INFO_ENDL();
    PRINT_INFO("vol_lab="); PRINT_INFO(fs_bpb_vol_lab); PRINT_INFO_ENDL();
    PRINT_INFO("fil_sys_type="); PRINT_INFO(fs_bpb_fil_sys_type); PRINT_INFO_ENDL();
    PRINT_INFO("fs_valid_fat_entries=0x"); PRINT_INFO_HU32(fs_valid_fat_entries); PRINT_INFO_ENDL();
    PRINT_INFO("fs_dir_ent_per_clus=0x"); PRINT_INFO_HU16(fs_dir_ent_per_clus); PRINT_INFO_ENDL();
    PRINT_INFO("fs_bytes_per_clus=0x"); PRINT_INFO_HU32(fs_bytes_per_clus); PRINT_INFO_ENDL();

    if(strncmp(fs_bpb_fil_sys_type, "FAT32   ", 8) != 0)
    {
        rv = -5;
        goto err_exit;
    }

    if(fs_bpb_bytes_per_sec != SEC_SIZE)
    {
        rv = -6;
        goto err_exit;
    }

err_exit:
    return rv;
}

int8_t fs_write_root_dir(void)
{
    uint32_t file_clus;
    uint32_t write_sec = data_sec(); // top of data sector
    uint32_t write_end_sec = write_sec + fs_log_cfg.root_dir_clusters * fs_bpb_sec_per_clus;
    dir_t* dir = (dir_t*)fs_buf;
    uint32_t i;
    int8_t rv = 0;

    memset(fs_buf, 0, sizeof(fs_buf));

    PRINT_DEBUG("\r\nfs_write_root_dir\r\n");
    memcpy(dir->Name, LOGGER_VOL_LAB, 11);
    dir->Attr = DIR_ATTR_VOLUME_ID;
    // follows windows created volume label
    fs_get_cur_datetime(&dir->WrtDate, &dir->WrtDate, NULL);
    dir++;

    file_clus = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters;
    memcpy(dir->Name, "LOG     CFG", 11);
    dir->Attr = DIR_ATTR_SYSTEM | DIR_ATTR_HIDDEN;
    fs_get_cur_datetime(&dir->CrtDate, &dir->CrtTime, &dir->CrtTimeTenth);
    dir->LstAccDate = dir->CrtDate;
    dir->WrtDate = dir->CrtDate;
    dir->WrtTime = dir->CrtTime;
    dir->FstClusHI = (uint16_t)(file_clus >> 16);
    dir->FstClusLO = (uint16_t)file_clus;
    dir->FileSize = sizeof(log_cfg_t);
    dir++;

    for(i = 0; i < fs_log_cfg.max_log_files; i++)
    {
        memset(dir, 0, sizeof(dir_t));
        strncpy(dir->Name, "        LOG", 11);
        u32tos2(dir->Name, i, fs_log_cfg.filename_digit, 0);
        dir->Attr = DIR_ATTR_SYSTEM | DIR_ATTR_HIDDEN;
        fs_get_cur_datetime(&dir->CrtDate, &dir->CrtTime, &dir->CrtTimeTenth);
        dir->LstAccDate = dir->CrtDate;
        dir->WrtDate = dir->CrtDate;
        dir->WrtTime = dir->CrtTime;
        dir++;

        if(dir >= (dir_t*)(fs_buf + sizeof(fs_buf)))
        {
            PRINT_DEBUG("file_no="); PRINT_DEBUG_U32(i); PRINT_DEBUG_ENDL();
            rv = sdc_write_block(fs_buf, write_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -1;
                goto err_exit;
            }
            dir = (dir_t*)fs_buf;
        }
    }

    if(dir > (dir_t*)(fs_buf))
    {
        // Full the rest of entries with zero.
        memset(dir, 0, (size_t)(fs_buf + sizeof(fs_buf) - (uint8_t*)dir));

        PRINT_DEBUG("file_no="); PRINT_DEBUG_U32(i); PRINT_DEBUG_ENDL();
        rv = sdc_write_block(fs_buf, write_sec++, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -2;
            goto err_exit;
        }
    }

    // Write the rest of sectors in the cluster.
    memset(fs_buf, 0, sizeof(fs_buf));
    while(write_sec < write_end_sec)
    {
        PRINT_DEBUG("write_sec="); PRINT_DEBUG_HU32(write_sec); PRINT_DEBUG_ENDL();
        rv = sdc_write_block(fs_buf, write_sec++, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -3;
            goto err_exit;
        }
    }

err_exit:
    return rv;
}

int8_t fs_write_log_cfg(void)
{
    uint32_t write_sec = log_cfg_sec();
    int8_t rv = 0;

    PRINT_DEBUG("\r\nfs_write_log_cfg\r\n");
    memset(fs_buf, 0, sizeof(fs_buf));
    memcpy(fs_buf, &fs_log_cfg, sizeof(fs_log_cfg));
    PRINT_DEBUG("max_log_files=0x"); PRINT_DEBUG_HU32(fs_log_cfg.max_log_files); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("root_dir_clusters=0x"); PRINT_DEBUG_HU32(fs_log_cfg.root_dir_clusters); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("start_log_file_no=0x"); PRINT_DEBUG_HU32(fs_log_cfg.start_log_file_no); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("current_log_files=0x"); PRINT_DEBUG_HU32(fs_log_cfg.current_log_files); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("log_file_clusters=0x"); PRINT_DEBUG_HU32(fs_log_cfg.log_file_clusters); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("fs_next_free_clus=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_next_free_clus); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("fs_free_clusters=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_free_clusters); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("filename_digit=0x"); PRINT_DEBUG_HU8(fs_log_cfg.filename_digit); PRINT_DEBUG_ENDL();
    rv = sdc_write_block(fs_buf, write_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }

err_exit:
    return rv;
}

int8_t fs_write_fat_entries()
{
    uint32_t write_sec = fat_sec(); // FAT0 sector
    uint32_t* fat_ent = (uint32_t*)fs_buf;
    uint32_t next_clus;
    uint32_t i;
    uint32_t read_sec;
    uint32_t end_clus;
    uint8_t j;
    int8_t rv = 0;

    PRINT_DEBUG("\r\nfs_write_fat_entries\r\n");
    // Reserved FAT entries
    *fat_ent++ = FAT0;
    for(i = 1; i < fs_bpb_root_clus; i++)
    {
        *fat_ent++ = FAT_EOC;
    }
    // root dir
    next_clus = fs_bpb_root_clus + 1;
    end_clus = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters;
    for(; i < end_clus; i++)
    {
        *fat_ent++ = ((i + 1) < fs_log_cfg.root_dir_clusters) ? next_clus++: FAT_EOC;
        if(fat_ent >= (uint32_t*)(fs_buf + sizeof(fs_buf)))
        {
            PRINT_DEBUG("root_dir_entry i=0x"); PRINT_DEBUG_HU32(i);
            PRINT_DEBUG(" next_clus=0x"); PRINT_DEBUG_HU32(next_clus); PRINT_DEBUG_ENDL();
            rv = sdc_write_block(fs_buf, write_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -1;
                goto err_exit;
            }
            fat_ent = (uint32_t*)fs_buf;
        }
    }
    // LOG.CFG
    *fat_ent++ = FAT_EOC;
    if(fat_ent >= (uint32_t*)(fs_buf + sizeof(fs_buf)))
    {
        PRINT_DEBUG("LOG.CFG\r\n");
        rv = sdc_write_block(fs_buf, write_sec++, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -2;
            goto err_exit;
        }
        fat_ent = (uint32_t*)fs_buf;
    }
    // log files
    end_clus = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + fs_log_cfg.log_file_clusters;
    for(; i < end_clus; i++)
    {
        *fat_ent++ = FAT_BAD_CLUSTER;
        if(fat_ent >= (uint32_t*)(fs_buf + sizeof(fs_buf)))
        {
            PRINT_DEBUG("log_file_ent i=0x"); PRINT_DEBUG_HU32(i); PRINT_DEBUG_ENDL();
            rv = sdc_write_block(fs_buf, write_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -3;
                goto err_exit;
            }
            fat_ent = (uint32_t*)fs_buf;
        }
    }
    if(fat_ent > (uint32_t*)fs_buf)
    {
        PRINT_DEBUG("log_file_ent i=0x"); PRINT_DEBUG_HU32(i); PRINT_DEBUG_ENDL();
        rv = sdc_write_block(fs_buf, write_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -4;
            goto err_exit;
        }
        fat_ent = (uint32_t*)fs_buf;
    }
    // Free the rest of clusters
    end_clus = fs_valid_fat_entries;
    for(; i < end_clus; i++)
    {
        *fat_ent++ = 0;
        if(fat_ent >= (uint32_t*)(fs_buf + sizeof(fs_buf)))
        {
            PRINT_DEBUG("free_ent i=0x"); PRINT_DEBUG_HU32(i); PRINT_DEBUG_ENDL();
            rv = sdc_write_block(fs_buf, write_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -3;
                goto err_exit;
            }
            fat_ent = (uint32_t*)fs_buf;
        }
    }
    if(fat_ent > (uint32_t*)fs_buf)
    {
        PRINT_DEBUG("free_ent i=0x"); PRINT_DEBUG_HU32(i); PRINT_DEBUG_ENDL();
        rv = sdc_write_block(fs_buf, write_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -4;
            goto err_exit;
        }
        fat_ent = (uint32_t*)fs_buf;
    }

    // Copy FAT0 to other FATs
    for(j = 1; j < fs_bpb_num_fats; j++)
    {
        read_sec = fat_sec(); // FAT0 sector
        write_sec = read_sec + j * fs_bpb_fat_sz32;  // FATn sector
        for(i = 0; i < fs_bpb_fat_sz32; i++)
        {
            PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32(i); PRINT_DEBUG_ENDL();
            rv = sdc_read_block(fs_buf, read_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -5;
                goto err_exit;
            }
            rv = sdc_write_block(fs_buf, write_sec++, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -6;
                goto err_exit;
            }
        }
    }

err_exit:
    return rv;
}

int8_t fs_write_fs_info()
{
    uint32_t acc_sec = fs_sec();
    int8_t rv;

    PRINT_DEBUG("\r\nfs_write_fs_info\r\n");
    rv = sdc_read_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }
    // FSI_LeadSig
    if(*((uint32_t*)&fs_buf[0]) != 0x41615252)
    {
        rv = -2;
        goto err_exit;
    }
    // FSI_StrucSig
    if(*((uint32_t*)&fs_buf[484]) != 0x61417272)
    {
        rv = -3;
        goto err_exit;
    }
    // FSI_TrailSig
    if(*((uint32_t*)&fs_buf[508]) != 0xAA550000)
    {
        rv = -4;
        goto err_exit;
    }
    // FSI_Nxt_Free
    *((uint32_t*)&fs_buf[492]) = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters +
            LOG_CFG_CLUSTERS + fs_log_cfg.log_file_clusters;
    // FSI_Free_Count
    *((uint32_t*)&fs_buf[488]) = fs_valid_fat_entries - *((uint32_t*)&fs_buf[492]);
    PRINT_DEBUG("FSI_Nxt_Free=0x"); PRINT_DEBUG_HU32(*((uint32_t*)&fs_buf[492])); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("FSI_Free_Count=0x"); PRINT_DEBUG_HU32(*((uint32_t*)&fs_buf[488])); PRINT_DEBUG_ENDL();

    rv = sdc_write_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -4;
        goto err_exit;
    }

err_exit:
    return rv;
}

int8_t set_volume_label(char* vol_lab)
{
    int8_t rv;
    uint32_t acc_sec = fs_bpb_sec;

    PRINT_DEBUG("\r\nset_volume_label\r\n");
    rv = sdc_read_block(fs_buf, fs_bpb_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }
    strcpy(fs_bpb_vol_lab, vol_lab);
    memcpy(fs_buf + 71, vol_lab, 11);

    // Write BPB
    rv = sdc_write_block(fs_buf, fs_bpb_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -2;
        goto err_exit;
    }

    // Write backup BPB
    acc_sec += fs_bpb_bk_boot_sec;
    rv = sdc_write_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -2;
        goto err_exit;
    }

err_exit:
    return rv;
}

int8_t fs_write_init_data(uint32_t max_files, uint32_t unmanaged_clusters)
{
    int8_t rv;
    PRINT_DEBUG("\r\nfs_write_init_data\r\n");

    if(max_files >= 100000000)
    {
        PRINT_ERROR("max_files is less than 100000000\r\n");
        rv = -1;
        goto err_exit;
    }

    fs_log_cfg.signature = LOG_CFG_SIG;
    fs_log_cfg.max_log_files = max_files;
    fs_log_cfg.start_log_file_no = 0;
    fs_log_cfg.current_log_files = 0;
    fs_log_cfg.root_dir_clusters = (2 + max_files + fs_dir_ent_per_clus - 1) / fs_dir_ent_per_clus; // Volume label, LOG.CFG, log files
    if((fs_valid_fat_entries - fs_bpb_root_clus - fs_log_cfg.root_dir_clusters) <= unmanaged_clusters)
    {
        PRINT_ERROR("unmanaged_clusters is too large, less than 0x");
        PRINT_ERROR_HU32(fs_valid_fat_entries - fs_bpb_root_clus - fs_log_cfg.root_dir_clusters);
        rv = -2;
        PRINT_ERROR("\r\n");
        goto err_exit;
    }
    fs_log_cfg.log_file_clusters = fs_valid_fat_entries - fs_bpb_root_clus
                - fs_log_cfg.root_dir_clusters - LOG_CFG_CLUSTERS - unmanaged_clusters;
    fs_log_cfg.fs_next_free_clus = fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS;
    fs_log_cfg.fs_free_clusters = fs_log_cfg.log_file_clusters;
    fs_log_cfg.filename_digit = u32tos2(fs_filename_buf, max_files - 1, 0, 0);

    rv = fs_write_root_dir();
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -3;
        goto err_exit;
    }
    rv = fs_write_log_cfg();
    if(rv != 0)
    {
        PRINT_ERROR("fs_write_log_cfg failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -4;
        goto err_exit;
    }
    rv = fs_write_fat_entries();
    if(rv != 0)
    {
        PRINT_ERROR("fs_write_fat_entries failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -5;
        goto err_exit;
    }
    rv = fs_write_fs_info();
    if(rv != 0)
    {
        PRINT_ERROR("fs_write_fs_info failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -6;
        goto err_exit;
    }
    rv = set_volume_label(LOGGER_VOL_LAB);
    if(rv != 0)
    {
        PRINT_ERROR("fs_write_fs_info failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -7;
        goto err_exit;
    }

err_exit:
    return rv;
}

int8_t fs_read_log_cfg(void)
{
    uint32_t read_sec = data_sec();
    dir_t* dir;
    int8_t rv = 0;

    PRINT_DEBUG("\r\nfs_read_log_cfg\r\n");
    // Read first sector of root dir
    rv = sdc_read_block(fs_buf, read_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }
    dir = (dir_t*)fs_buf;
    dir++; // entry #1

    // Read LOG.CFG
    read_sec = clus_to_sec((uint32_t)dir->FstClusHI << 16 | dir->FstClusLO);
    rv = sdc_read_block(fs_buf, read_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -1;
        goto err_exit;
    }
    if(((log_cfg_t*)fs_buf)->signature != LOG_CFG_SIG)
    {
        PRINT_ERROR("CFG.LOG signature is invalid (0x"); PRINT_ERROR_U32(((log_cfg_t*)fs_buf)->signature); PRINT_ERROR(")\r\n");
        rv = -2;
        goto err_exit;
    }
    memcpy(&fs_log_cfg, fs_buf, sizeof(fs_log_cfg));

    PRINT_INFO("max_log_files=0x"); PRINT_INFO_HU32(fs_log_cfg.max_log_files); PRINT_INFO_ENDL();
    PRINT_INFO("root_dir_clusters=0x"); PRINT_INFO_HU32(fs_log_cfg.root_dir_clusters); PRINT_INFO_ENDL();
    PRINT_INFO("start_log_file_no=0x"); PRINT_INFO_HU32(fs_log_cfg.start_log_file_no); PRINT_INFO_ENDL();
    PRINT_INFO("current_log_files=0x"); PRINT_INFO_HU32(fs_log_cfg.current_log_files); PRINT_INFO_ENDL();
    PRINT_INFO("log_file_clusters=0x"); PRINT_INFO_HU32(fs_log_cfg.log_file_clusters); PRINT_INFO_ENDL();
    PRINT_INFO("fs_next_free_clus=0x"); PRINT_INFO_HU32(fs_log_cfg.fs_next_free_clus); PRINT_INFO_ENDL();
    PRINT_INFO("fs_free_clusters=0x"); PRINT_INFO_HU32(fs_log_cfg.fs_free_clusters); PRINT_INFO_ENDL();
    PRINT_INFO("filename_digit=0x"); PRINT_INFO_HU8(fs_log_cfg.filename_digit); PRINT_INFO_ENDL();

err_exit:
   return rv;
}

int8_t fs_init(uint32_t max_files, uint32_t unmanaged_clusters, uint8_t force_init)
{
    int8_t rv;

    PRINT_DEBUG("\r\nfs_init\r\n");
    if(fs_initialized)
    {
        PRINT_ERROR("already initialized\r\n");
        rv = -1;
        goto err_exit;
    }

    rv = fs_get_mbr_bpb();
    if(rv != 0)
    {
        PRINT_ERROR("fs_get_mbr_bpb failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -2;
        goto err_exit;
    }

    if(force_init || strncmp(fs_bpb_vol_lab, LOGGER_VOL_LAB, 11) != 0)
    {
        rv = fs_write_init_data(max_files, unmanaged_clusters);
        if(rv != 0)
        {
            PRINT_ERROR("fs_write_init_data failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -3;
            goto err_exit;
        }
    }
    else
    {
        rv = fs_read_log_cfg();
        if(rv != 0)
        {
            PRINT_ERROR("fs_read_log_cfg failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -4;
            goto err_exit;
        }
    }

    fs_initialized = 1;

err_exit:
    return rv;
}

int8_t fs_delete_oldest_file(void)
{
    uint32_t dir_ent_index;
    uint32_t acc_sec;
    uint32_t fat_ent_last_sec;
    uint32_t file_clus;
    uint32_t file_clusters;
    uint32_t file_end_clus;
    dir_t* dir;
    int8_t rv = 0;
    uint8_t i;

    PRINT_DEBUG("\r\nfs_delete_oldest_file\r\n");
    // Delete file entries
    if(fs_log_cfg.current_log_files == 0)
    {
        PRINT_DEBUG("all files were deleted\r\n");
        return -1;
    }

    // Modify dir entry of the first log file
    dir_ent_index = log_file_no_to_dir_ent_index(fs_log_cfg.start_log_file_no);
    acc_sec = dir_ent_index_to_sec(dir_ent_index);
    PRINT_DEBUG("start_log_file_no=0x"); PRINT_DEBUG_HU32(fs_log_cfg.start_log_file_no); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("dir_ent_index=0x"); PRINT_DEBUG_HU32(dir_ent_index); PRINT_DEBUG_ENDL();
    rv = sdc_read_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -2;
        goto err_exit;
    }
    dir = (dir_t*)fs_buf;
    dir += dir_ent_index % DIR_ENT_PER_SEC;
    file_clus = (uint32_t)dir->FstClusHI << 16 | dir->FstClusLO;
    file_clusters = (dir->FileSize + fs_bytes_per_clus - 1) / fs_bytes_per_clus;
    dir->Attr = DIR_ATTR_SYSTEM | DIR_ATTR_HIDDEN;
    dir->FstClusHI = 0;
    dir->FstClusLO = 0;
    dir->FileSize = 0;
    PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("file_clusters=0x"); PRINT_DEBUG_HU32(file_clusters); PRINT_DEBUG_ENDL();
    rv = sdc_write_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -3;
        goto err_exit;
    }

    // Modify LOG.CFG
    fs_log_cfg.current_log_files--;
    fs_log_cfg.start_log_file_no++;
    if(fs_log_cfg.start_log_file_no >= fs_log_cfg.max_log_files) fs_log_cfg.start_log_file_no = 0;
    fs_log_cfg.fs_free_clusters += file_clusters;
    rv = fs_write_log_cfg();
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -4;
        goto err_exit;
    }

    // Write FAT entries
    file_end_clus = log_file_clus_add(file_clus, file_clusters); // may wrap around
    for(i = 0; i < fs_bpb_num_fats; i++)
    {
        uint32_t file_clus0 = file_clus;
        uint32_t* fat_ent;
        uint32_t next_acc_sec;

        acc_sec = fat_ent_to_sec(file_clus0, i);
        rv = sdc_read_block(fs_buf, acc_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -1;
            goto err_exit;
        }
        fat_ent = (uint32_t*)fs_buf + (file_clus0 % FAT_ENT_PER_SEC);
        PRINT_DEBUG("read_sec0=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
        PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
        PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();

        while(file_clus0 != file_end_clus)
        {
            uint32_t cur_file_clus = file_clus0;
            file_clus0 = log_file_clus_add(file_clus0, 1);
            *fat_ent++ = FAT_BAD_CLUSTER;
            next_acc_sec = fat_ent_to_sec(file_clus0, i);
            if(acc_sec != next_acc_sec)
            {
                rv = sdc_write_block(fs_buf, acc_sec, 1);
                if(rv != 0)
                {
                    PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                    rv = -2;
                    goto err_exit;
                }
                PRINT_DEBUG("write_sec1=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(cur_file_clus); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
                acc_sec = next_acc_sec;
                rv = sdc_read_block(fs_buf, acc_sec, 1);
                if(rv != 0)
                {
                    PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                    rv = -3;
                    goto err_exit;
                }
                fat_ent = (uint32_t*)fs_buf + (file_clus0 % FAT_ENT_PER_SEC);
                PRINT_DEBUG("read_sec2=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
            }
        }
        if(acc_sec == next_acc_sec)
        {
            rv = sdc_write_block(fs_buf, acc_sec, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -2;
                goto err_exit;
            }
            PRINT_DEBUG("write_sec3=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
            PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
            PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
        }
    }

err_exit:
    return rv;
}

int8_t fs_open(uint32_t reserved_file_size)
{
    int8_t rv = 0;
    uint32_t reserved_clusters;

    PRINT_ERROR("\r\nfs_open\r\n");
    if(!fs_initialized)
    {
        PRINT_ERROR("FSYS is not initialized\r\n");
        rv = -1;
        goto err_exit;
    }
    if(fs_file_opened)
    {
        PRINT_ERROR("File is opened\r\n");
        rv = -2;
        goto err_exit;
    }

    reserved_clusters = (reserved_file_size + fs_bytes_per_clus - 1) / fs_bytes_per_clus;
    PRINT_DEBUG("reserved_clusters=0x"); PRINT_DEBUG_HU32(reserved_clusters); PRINT_DEBUG_ENDL();
    while(reserved_clusters > fs_log_cfg.fs_free_clusters || fs_log_cfg.current_log_files >= fs_log_cfg.max_log_files)
    {
        rv = fs_delete_oldest_file();
        if(rv != 0)
        {
            PRINT_ERROR("fs_delete_old_files failed\r\n");
            rv = -3;
            goto err_exit;
        }
    }

    PRINT_DEBUG("fs_next_free_clus=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_next_free_clus); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("fs_free_clusters=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_free_clusters); PRINT_DEBUG_ENDL();
    fs_log_write_sec = clus_to_sec(fs_log_cfg.fs_next_free_clus);
    fs_log_cached_size = 0;
    fs_log_file_size = 0;
    fs_file_opened = 1;
    PRINT_DEBUG("fs_log_write_sec=0x"); PRINT_DEBUG_HU32(fs_log_write_sec); PRINT_DEBUG_ENDL();

err_exit:
    return rv;
}

int8_t fs_write(uint8_t* buff, uint16_t length)
{
    int8_t rv = 0;
    uint16_t remaining_size;
    uint32_t log_file_start_sec = clus_to_sec(fs_bpb_root_clus + fs_log_cfg.root_dir_clusters + LOG_CFG_CLUSTERS);
    uint32_t log_file_end_sec = clus_to_sec(fs_bpb_root_clus + fs_log_cfg.root_dir_clusters +
                LOG_CFG_CLUSTERS + fs_log_cfg.log_file_clusters);

    PRINT_DEBUG("\r\nfs_write\r\n");
    PRINT_DEBUG("log_file_start_sec=0x"); PRINT_DEBUG_HU32(log_file_start_sec); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("log_file_end_sec=0x"); PRINT_DEBUG_HU32(log_file_end_sec); PRINT_DEBUG_ENDL();

    if(!fs_initialized)
    {
        PRINT_ERROR("FSYS is not initialized\r\n");
        rv = -1;
        goto err_exit;
    }
    if(!fs_file_opened)
    {
        PRINT_ERROR("File is not opened\r\n");
        rv = -2;
        goto err_exit;
    }

    remaining_size = length;

    while(remaining_size > 0)
    {
        uint32_t cache_cp_size;

        // Update cache
        if((remaining_size + fs_log_cached_size) < sizeof(fs_buf))
        {
            // Only cache the data
            memcpy(fs_buf + fs_log_cached_size, buff + length - remaining_size, remaining_size);
            fs_log_cached_size += remaining_size;
            remaining_size = 0;
            break;
        }

        cache_cp_size = sizeof(fs_buf) - fs_log_cached_size;
        memcpy(fs_buf + fs_log_cached_size, buff + length - remaining_size, (size_t)cache_cp_size);
        fs_log_cached_size = 0;
        remaining_size -= cache_cp_size;
        fs_log_file_size += sizeof(fs_buf);
        //PRINT_DEBUG("remaining_size=0x"); PRINT_DEBUG_HU32(remaining_size); PRINT_DEBUG_ENDL();

        // Write cache
        rv = sdc_write_block(fs_buf, fs_log_write_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -4;
            goto err_exit;
        }
        if(++fs_log_write_sec >= log_file_end_sec)
        {
            fs_log_write_sec = log_file_start_sec;
        }
        //if((fs_log_file_size & 0xffffffUL) == 0)
        //{
            PRINT_DEBUG("fs_log_write_sec=0x"); PRINT_DEBUG_HU32(fs_log_write_sec); PRINT_DEBUG_ENDL();
        //}
    }
    //if((fs_log_file_size & 0xffffffUL) == 0)
    //{
        PRINT_DEBUG("fs_log_file_size=0x"); PRINT_DEBUG_HU32(fs_log_file_size); PRINT_DEBUG_ENDL();
        PRINT_DEBUG("fs_log_cached_size=0x"); PRINT_DEBUG_HU32(fs_log_cached_size); PRINT_DEBUG_ENDL();
    //}

err_exit:
    return rv;
}

int8_t fs_close(void)
{
    int8_t rv = 0;
    uint32_t acc_sec;
    uint32_t file_clusters;
    uint32_t dir_ent_index;
    uint32_t file_end_clus;
    uint32_t file_no;
    dir_t* dir;
    uint8_t i;

    PRINT_DEBUG("\r\nfs_close\r\n");

    if(!fs_initialized)
    {
        PRINT_ERROR("FSYS is not initialized\r\n");
        rv = -1;
        goto err_exit;
    }
    if(!fs_file_opened)
    {
        PRINT_ERROR("File is not opened\r\n");
        rv = -2;
        goto err_exit;
    }

    if(fs_log_cached_size > 0)
    {
        if((fs_log_file_size + SEC_SIZE + fs_bytes_per_clus - 1) / fs_bytes_per_clus
                >= fs_log_cfg.fs_free_clusters)
        {
            PRINT_ERROR("RESERVED.LOG is full\r\n");
            rv = -3;
            goto err_exit;
        }

        // fill reset of the sector with zero.
        memset(fs_buf + fs_log_cached_size, 0, sizeof(fs_buf) - fs_log_cached_size);

        // Write cache
        rv = sdc_write_block(fs_buf, fs_log_write_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -4;
            goto err_exit;
        }
        fs_log_file_size += fs_log_cached_size;
        PRINT_DEBUG("fs_log_write_sec=0x"); PRINT_DEBUG_HU32(fs_log_write_sec); PRINT_DEBUG_ENDL();
    }
    file_clusters = (fs_log_file_size + fs_bytes_per_clus - 1) / fs_bytes_per_clus;
    PRINT_DEBUG("fs_log_file_size=0x"); PRINT_DEBUG_HU32(fs_log_file_size); PRINT_DEBUG("\r\n");
    PRINT_DEBUG("file_clusters=0x"); PRINT_DEBUG_HU32(file_clusters); PRINT_DEBUG("\r\n");

    // Modify file entry
    file_no = fs_log_cfg.start_log_file_no + fs_log_cfg.current_log_files;
    if(file_no >= fs_log_cfg.max_log_files) file_no -= fs_log_cfg.max_log_files;
    fs_log_cfg.current_log_files++;

    dir_ent_index = log_file_no_to_dir_ent_index(file_no);
    acc_sec = dir_ent_index_to_sec(dir_ent_index);
    PRINT_DEBUG("start_log_file_no=0x"); PRINT_DEBUG_HU32(fs_log_cfg.start_log_file_no); PRINT_DEBUG("\r\n");
    PRINT_DEBUG("file_no=0x"); PRINT_DEBUG_HU32(file_no); PRINT_DEBUG("\r\n");
    PRINT_DEBUG("dir_ent_index=0x"); PRINT_DEBUG_HU32(dir_ent_index); PRINT_DEBUG("\r\n");
    rv = sdc_read_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -5;
        goto err_exit;
    }
    dir = (dir_t*)fs_buf;
    dir += dir_ent_index % DIR_ENT_PER_SEC;
    dir->Attr = DIR_ATTR_READ_ONLY;
    strncpy(dir->Name, "        LOG", 11); // in case name was broken
    u32tos2(dir->Name, file_no, fs_log_cfg.filename_digit, 0);
    dir->FstClusHI = (uint16_t)(fs_log_cfg.fs_next_free_clus >> 16);
    dir->FstClusLO = (uint16_t)fs_log_cfg.fs_next_free_clus;
    dir->FileSize = fs_log_file_size;
    fs_get_cur_datetime(&dir->CrtDate, &dir->CrtTime, &dir->CrtTimeTenth);
    dir->LstAccDate = dir->CrtDate;
    dir->WrtDate = dir->CrtDate;
    dir->WrtTime = dir->CrtTime;
    PRINT_DEBUG("fs_next_free_clus=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_next_free_clus); PRINT_DEBUG_ENDL();
    rv = sdc_write_block(fs_buf, acc_sec, 1);
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -6;
        goto err_exit;
    }

    // Modify FAT
    file_end_clus = log_file_clus_add(fs_log_cfg.fs_next_free_clus, file_clusters); // may wrap around
    for(i = 0; i < fs_bpb_num_fats; i++)
    {
        uint32_t file_clus0 = fs_log_cfg.fs_next_free_clus;
        uint32_t* fat_ent;
        uint32_t next_acc_sec;

        acc_sec = fat_ent_to_sec(file_clus0, i);
        rv = sdc_read_block(fs_buf, acc_sec, 1);
        if(rv != 0)
        {
            PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
            rv = -7;
            goto err_exit;
        }
        fat_ent = (uint32_t*)fs_buf + (file_clus0 % FAT_ENT_PER_SEC);
        PRINT_DEBUG("read_sec0=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
        PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
        PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();

        while(file_clus0 != file_end_clus)
        {
            uint32_t cur_file_clus = file_clus0;
            file_clus0 = log_file_clus_add(file_clus0, 1);
            *fat_ent++ = (file_clus0 == file_end_clus) ? FAT_EOC: file_clus0;
            next_acc_sec = fat_ent_to_sec(file_clus0, i);
            if(acc_sec != next_acc_sec)
            {
                rv = sdc_write_block(fs_buf, acc_sec, 1);
                if(rv != 0)
                {
                    PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                    rv = -8;
                    goto err_exit;
                }
                PRINT_DEBUG("write_sec1=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(cur_file_clus); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
                acc_sec = next_acc_sec;
                rv = sdc_read_block(fs_buf, acc_sec, 1);
                if(rv != 0)
                {
                    PRINT_ERROR("sdc_read_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                    rv = -9;
                    goto err_exit;
                }
                fat_ent = (uint32_t*)fs_buf + (file_clus0 % FAT_ENT_PER_SEC);
                PRINT_DEBUG("read_sec2=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
                PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
            }
        }
        if(acc_sec == next_acc_sec)
        {
            rv = sdc_write_block(fs_buf, acc_sec, 1);
            if(rv != 0)
            {
                PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
                rv = -2;
                goto err_exit;
            }
            PRINT_DEBUG("write_sec3=0x"); PRINT_DEBUG_HU32(acc_sec); PRINT_DEBUG_ENDL();
            PRINT_DEBUG("file_clus=0x"); PRINT_DEBUG_HU32(file_clus0); PRINT_DEBUG_ENDL();
            PRINT_DEBUG("fat_ent=0x"); PRINT_DEBUG_HU32((uint32_t)(fat_ent - (uint32_t*)fs_buf)); PRINT_DEBUG_ENDL();
        }
    }

    fs_log_cfg.fs_free_clusters -= file_clusters;
    fs_log_cfg.fs_next_free_clus = log_file_clus_add(fs_log_cfg.fs_next_free_clus, file_clusters);

    // Modify LOG.CFG
    rv = fs_write_log_cfg();
    if(rv != 0)
    {
        PRINT_ERROR("sdc_write_block failed ("); PRINT_ERROR_I8(rv); PRINT_ERROR(")\r\n");
        rv = -10;
        goto err_exit;
    }

    PRINT_DEBUG("fs_next_free_clus=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_next_free_clus); PRINT_DEBUG_ENDL();
    PRINT_DEBUG("fs_free_clusters=0x"); PRINT_DEBUG_HU32(fs_log_cfg.fs_free_clusters); PRINT_DEBUG_ENDL();

    fs_file_opened = 0;

err_exit:
    return rv;
}

void fs_deinit(void)
{
    fs_initialized = 0;
}
