// Copyright 2024 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>
#include "uart_util.h"
#include "uart1.h"
#include "uart2.h"
#include "utility.h"
#include "tmr.h"
#include "switch.h"
#include "lcd.h"
#include "chgv.h"
#include "common.h"
#include "spi.h"
#include "sdc.h"
#include "fsys.h"
#include "dcmd.h"

#define DAT_BUF_SIZE  512
#define DAT_BUF_COUNT 2
#define DCMD_BUF_SIZE 128
#define UNIT_NUM      4

typedef struct _dbg_cmd_t
{
    const char* cmd;
    void (*exec)(char* args);
} dbg_cmd_t;

// variables in mainc.c
extern uint8_t state;
extern uint8_t err_code;
extern uint8_t fs_st;
extern uint8_t fs_err;
extern uint8_t enable_print_st;

uint8_t msg_buf_len[UART_DEV_NUM];
char msg_buf[UART_DEV_NUM][DCMD_BUF_SIZE];
uint8_t data_buf[DAT_BUF_COUNT][DAT_BUF_SIZE];

#define CHGV8     "chgv8"
#define CHGV16    "chgv16"
#define CHGV24    "chgv24"
#define CHGST     "chgst"
#define ENPRINTST "enprintst"
#define FSINIT    "fsinit"
#define FSTEST    "fstest"
#define SDCI      "sdci"
#define SDCR      "sdcr"
#define SPITR     "spitr"
#define STATE     "state"

void dcmd_unit(char* args);
void dcmd_chgv8(char* args);
void dcmd_chgv16(char* args);
void dcmd_chgv24(char* args);
void dcmd_chgst(char* args);
void dcmd_enprintst(char* args);
void dcmd_fsinit(char* args);
void dcmd_fstest(char* args);
void dcmd_sdci(char* args);
void dcmd_sdcr(char* args);
void dcmd_spitr(char* args);
void dcmd_state(char* args);
void dcmd_uc(char* args);

dbg_cmd_t cmd_list[] =
{
    {CHGV8,     dcmd_chgv8},
    {CHGV16,    dcmd_chgv16},
    {CHGV24,    dcmd_chgv24},
    {CHGST,     dcmd_chgst},
    {ENPRINTST, dcmd_enprintst},
    {FSINIT,    dcmd_fsinit},
    {FSTEST,    dcmd_fstest},
    {SDCI,      dcmd_sdci},
    {SDCR,      dcmd_sdcr},
    {SPITR,     dcmd_spitr},
    {STATE,     dcmd_state},
};
#define CMD_COUNT (uint8_t)(sizeof(cmd_list) / sizeof(cmd_list[0]))

uint8_t unit = UNIT_NUM;

void dcmd_unit(char* args)
{
    uint8_t i;
    i = stou8(&unit, args);
    uart_print(UART_DEV_DBG, (i > 0) ? "ok ": "NG ");
    if(unit > UNIT_NUM) unit = UNIT_NUM;
    uart_print(UART_DEV_DBG, "unit "); uart_print_u8(UART_DEV_DBG, unit);
    uart_print_endl(UART_DEV_DBG);
    chgv_set_com_slot(unit);
}

uint8_t dcmd_getsubstr(char* str, uint8_t* substr_pos)
{
    uint8_t i = 0;
    for(; str[i] != 0; i++)
    {
        if(str[i] != ' ') break;
    }
    if(str[i] == 0) return 0;
    *substr_pos = i;
    for(; str[i] != 0; i++)
    {
        if(str[i] == ' ') break;
    }
    return (i - *substr_pos);
}

void dcmd_chgv8(char* args)
{
    int8_t rv = 0;
    uint8_t err = 0;
    uint8_t i, k;
    uint8_t slot;
    uint8_t var;
    uint8_t v;

    i = stou8(&slot, args);
    if(i == 0)
    {
        err = 1;
        goto err_exit;
    }

    k = dcmd_getsubstr(args + i, &var);
    if(k == 0)
    {
        err = 2;
        goto err_exit;
    }
    var += i;
    i = var + k;
    if(args[i] != 0) args[i++] = 0; // expect the next argument
    k = stou8(&v, args + i);
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    if(k == 0)
    {
        rv = chgv_read_config_u8(&v, args + var);
        if(rv != 0)
        {
            err = 3;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "read v="); uart_print_u8(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    else
    {
        rv = chgv_write_config_u8(args + var, v);
        if(rv != 0)
        {
            err = 4;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "write v="); uart_print_u8(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_chgv16(char* args)
{
    int8_t rv = 0;
    uint8_t err = 0;
    uint8_t i, k;
    uint8_t slot;
    uint8_t var;
    uint16_t v;

    i = stou8(&slot, args);
    if(i == 0)
    {
        err = 1;
        goto err_exit;
    }

    k = dcmd_getsubstr(args + i, &var);
    if(k == 0)
    {
        err = 2;
        goto err_exit;
    }
    var += i;
    i = var + k;
    if(args[i] != 0) args[i++] = 0; // expect the next argument
    k = stou16(&v, args + i);
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    if(k == 0)
    {
        rv = chgv_read_config_u16(&v, args + var);
        if(rv != 0)
        {
            err = 3;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "read v="); uart_print_u16(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    else
    {
        rv = chgv_write_config_u16(args + var, v);
        if(rv != 0)
        {
            err = 4;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "write v="); uart_print_u16(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_chgv24(char* args)
{
    int8_t rv = 0;
    uint8_t err = 0;
    uint8_t i, k;
    uint8_t slot;
    uint8_t var;
    __uint24 v;

    i = stou8(&slot, args);
    if(i == 0)
    {
        err = 1;
        goto err_exit;
    }

    k = dcmd_getsubstr(args + i, &var);
    if(k == 0)
    {
        err = 2;
        goto err_exit;
    }
    var += i;
    i = var + k;
    if(args[i] != 0) args[i++] = 0; // expect the next argument
    k = stou24(&v, args + i);
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    if(k == 0)
    {
        rv = chgv_read_config_u24(&v, args + var);
        if(rv != 0)
        {
            err = 3;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "read v="); uart_print_u24(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    else
    {
        rv = chgv_write_config_u24(args + var, v);
        if(rv != 0)
        {
            err = 4;
            goto err_exit;
        }
        uart_print(UART_DEV_DBG, "write v="); uart_print_u24(UART_DEV_DBG, v);
        uart_print_endl(UART_DEV_DBG); uart_print(UART_DEV_DBG, "ok");
    }
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_read_status_cb(uint8_t slot, char* status_str)
{
    uart_print(UART_DEV_DBG, "slot ");  uart_print_u8(UART_DEV_DBG, slot); uart_print_endl(UART_DEV_DBG);
    uart_print(UART_DEV_DBG, status_str);
}

void dcmd_chgst(char* args)
{
    uint8_t i;
    uint8_t slot = 0;
    i = stou8(&slot, args);
    if(i > 0)
    {
        chgv_read_status(slot, dcmd_read_status_cb);
    }
    else
    {
        uart_print(UART_DEV_DBG, "ng "); uart_print_endl(UART_DEV_DBG);
    }
}

void dcmd_enprintst(char* args)
{
    uint8_t i;
    i = stou8(&enable_print_st, args);
    uart_print(UART_DEV_DBG, (i > 0) ? "ok ": "NG ");
    uart_print_u8(UART_DEV_DBG, enable_print_st); uart_print_endl(UART_DEV_DBG);
}

void dcmd_fsinit(char* args)
{
    int8_t rv = 0;
    uint8_t j, k;
    uint32_t v0;
    uint32_t v1;
    uint8_t v2;
    uint8_t err = 0;

    j = 0;
    k = stou32(&v0, args);
    if(k == 0)
    {
        err = 1;
        goto err_exit;
    }
    j += k;
    k = stou32(&v1, args);
    if(k == 0)
    {
        err = 1;
        goto err_exit;
    }
    j += k;
    k = stou8(&v2, args + j);
    if(k == 0)
    {
        err = 2;
        goto err_exit;
    }

    rv = sdc_init();
    if(rv != 0)
    {
        err = 3;
        goto err_exit;
    }

    rv = fs_init(v0, v1, v2);
    if(rv != 0)
    {
        err = 4;
        goto err_exit;
    }
    uart_print(UART_DEV_DBG, "ok");
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

int8_t cmd_fstest0(void)
{
    int8_t rv;

    uart_print(UART_DEV_DBG, "test#0\r\n");
    rv = fs_init(10000, 0x72D00, 1);
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "fs_init failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
        return -1;
    }

    return 0;
}

int8_t cmd_fstest1(void)
{
    int8_t rv;
    uint16_t i;

    uart_print(UART_DEV_DBG, "test#1\r\n");

    rv = fs_init(10, 0x72D00, 1);
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "fs_init failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
        return -1;
    }

    for(i = 0; i < 12; i++)
    {
        uint32_t file_size = 10485760UL; // 10MB
        uint32_t write_size;
        uint32_t j;

        rv = fs_open(file_size);
        if(rv != 0)
        {
            uart_print(UART_DEV_DBG, "fs_open failed rv=");
            uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
            return -2;
        }

        write_size = file_size - i;
        uart_print(UART_DEV_DBG, "write size=");
        uart_print_u32(UART_DEV_DBG, write_size); uart_print_endl(UART_DEV_DBG);
        j = 0;
        while(write_size > 0)
        {
            uint32_t k;
            uint16_t write_size0 = (write_size > DAT_BUF_SIZE) ? DAT_BUF_SIZE: (uint16_t)write_size;

            for(k = 0; k < write_size0; k += 8)
            {
                u32tohs2((char*)&data_buf[0][k], j + k, 0);
            }
            j += k;

            rv = fs_write(data_buf[0], write_size0);
            if(rv != 0)
            {
                uart_print(UART_DEV_DBG, "fs_write failed rv=");
                uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
                return -3;
            }

            write_size -= write_size0;
        }

        rv = fs_close();
        if(rv != 0)
        {
            uart_print(UART_DEV_DBG, "fs_close failed rv=");
            uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
            return -4;
        }
    }

    return 0;
}

int8_t cmd_fstest2(void)
{
    int8_t rv;
    uint16_t i;
    uint32_t file_size = 10485760UL; // 10MB
    uint32_t write_size;
    uint32_t j;

    uart_print(UART_DEV_DBG, "test#2 create 1 files\r\n");

    rv = fs_open(file_size);
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "fs_open failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
        return -1;
    }

    write_size = file_size;
    uart_print(UART_DEV_DBG, "write size=");
    uart_print_u32(UART_DEV_DBG, write_size); uart_print_endl(UART_DEV_DBG);
    j = 0;
    while(write_size > 0)
    {
        uint32_t k;
        uint16_t write_size0 = (write_size > DAT_BUF_SIZE) ? DAT_BUF_SIZE: (uint16_t)write_size;

        for(k = 0; k < write_size0; k += 8)
        {
            u32tohs2((char*)&data_buf[0][k], j + k, 0);
        }
        j += k;

        rv = fs_write(data_buf[0], write_size0);
        if(rv != 0)
        {
            uart_print(UART_DEV_DBG, "fs_write failed rv=");
            uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
            return -2;
        }

        write_size -= write_size0;
    }

    rv = fs_close();
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "fs_close failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
        return -3;
    }

    return 0;
}

int8_t cmd_fstest3(void)
{
    int8_t rv;
    uint16_t i;

    uart_print(UART_DEV_DBG, "test#3\r\n");

    rv = fs_init(16, 0x72D00, 1);
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "fs_init failed rv=");
        uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
        return -1;
    }

    for(i = 0; i < 13; i++)
    {
        uint32_t file_size = 10485760UL; // 10MB
        uint32_t write_size;
        uint32_t j;

        rv = fs_open(file_size);
        if(rv != 0)
        {
            uart_print(UART_DEV_DBG, "fs_open failed rv=");
            uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
            return -2;
        }

        write_size = file_size - i;
        uart_print(UART_DEV_DBG, "write size=");
        uart_print_u32(UART_DEV_DBG, write_size); uart_print_endl(UART_DEV_DBG);
        j = 0;
        while(write_size > 0)
        {
            uint32_t k;
            uint16_t write_size0 = (write_size > DAT_BUF_SIZE) ? DAT_BUF_SIZE: (uint16_t)write_size;

            for(k = 0; k < write_size0; k += 8)
            {
                u32tohs2((char*)&data_buf[0][k], j + k, 0);
            }
            j += k;

            rv = fs_write(data_buf[0], write_size0);
            if(rv != 0)
            {
                uart_print(UART_DEV_DBG, "fs_write failed rv=");
                uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
                return -3;
            }

            write_size -= write_size0;
        }

        rv = fs_close();
        if(rv != 0)
        {
            uart_print(UART_DEV_DBG, "fs_close failed rv=");
            uart_print_i8(UART_DEV_DBG, rv); uart_print_endl(UART_DEV_DBG);
            return -4;
        }
    }

    return 0;
}

void dcmd_fstest(char* args)
{
    uint8_t j;
    uint8_t v;
    uint8_t err = 0;
    int8_t rv = 0;

    j = htou8(&v, args);
    if(j == 0)
    {
        err = 1;
        goto err_exit;
    }

    if(v == 0)
    {
        rv = cmd_fstest0();
        if(rv != 0)
        {
            err = 2;
            goto err_exit;
        }
    }
    else if(v == 1)
    {
        rv = cmd_fstest1();
        if(rv != 0)
        {
            err = 3;
            goto err_exit;
        }
    }
    else if(v == 2)
    {
        rv = cmd_fstest2();
        if(rv != 0)
        {
            err = 4;
            goto err_exit;
        }
    }
    else if(v == 3)
    {
        rv = cmd_fstest3();
        if(rv != 0)
        {
            err = 5;
            goto err_exit;
        }
    }
    uart_print(UART_DEV_DBG, "ok");

    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_sdci(char* args)
{
    (void)args;
    int8_t rv;

    rv = sdc_init();
    if(rv != 0)
    {
        uart_print(UART_DEV_DBG, "ng rv "); uart_print_i8(UART_DEV_DBG, rv);
    }
    else
    {
        uart_print(UART_DEV_DBG, "ok");
    }
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_sdcr(char* args)
{
    uint32_t v0;
    uint16_t v1;
    uint8_t v2;
    uint8_t j, k;
    int8_t rv = 0;
    uint8_t err = 0;

    // sdrd [start address] [block count] [binary mode]
    j = 0;
    k = htou32(&v0, args);
    if(k == 0)
    {
        err = 1;
        goto err_exit;
    }
    j += k;
    k = htou16(&v1, args + j);
    if(k == 0)
    {
        err = 2;
        goto err_exit;
    }
    j += k;
    k = htou8(&v2, args + j);
    if(k == 0)
    {
        err = 3;
        goto err_exit;
    }
    j += k;
    rv = sdc_read_block(data_buf[0], v0, v1);
    if(rv != 0)
    {
        err = 4; uart_print_i8(UART_DEV_DBG, rv);
        goto err_exit;
    }
    uart_print(UART_DEV_DBG, "ok\r\n");
    if(v2 == 1)
    {
        uart1_send(data_buf[0], v1 * 512, 1);
    }
    else
    {
        uart_print_bytes(UART_DEV_DBG, data_buf[0], v1 * 512);
    }
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_setcfg(char* args)
{
    uint8_t j;
    uint8_t v;
    int8_t rv = 0;
    uint8_t err = 0;

    j = htou8(&v, args);
    if (j == 0)
    {
        err = 1;
        goto err_exit;
    }

    rv = chgv_write_config(v);
    if(rv != 0)
    {
        err = 2; uart_print_i8(UART_DEV_DBG, rv);
        goto err_exit;
    }
    uart_print(UART_DEV_DBG, "ok\r\n");
    uart_print(UART_DEV_DBG, "slot"); uart_print_hu8(UART_DEV_DBG, v);
    uart_print_endl(UART_DEV_DBG);
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
    uart_print(UART_DEV_DBG, " rv"); uart_print_i8(UART_DEV_DBG, rv);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_spitr(char* args)
{
    uint16_t v1;
    uint8_t err = 0;

    if(strlen(args) > (DAT_BUF_SIZE * 2))
    {
        err = 1;
        goto err_exit;
    }

    stobytes2(data_buf[0], sizeof(data_buf[0]), args, &v1);
    if(v1 == 0)
    {
        err = 2;
        goto err_exit;
    }
    spi_set_cs(0);
    spi_transfer(data_buf[0], data_buf[1], v1);
    spi_set_cs(1);
    uart_print(UART_DEV_DBG, "ok\r\n");
    uart_print_bytes(UART_DEV_DBG, data_buf[0], v1); uart_print_endl(UART_DEV_DBG);
    uart_print_bytes(UART_DEV_DBG, data_buf[1], v1); uart_print_endl(UART_DEV_DBG);
    goto end;

err_exit:
    uart_print(UART_DEV_DBG, "ng "); uart_print_u8(UART_DEV_DBG, err);
end:
    uart_print_endl(UART_DEV_DBG);
}

void dcmd_state(char* args)
{
    (void)args;
    uart_print(UART_DEV_DBG, "state="); uart_print_u8(UART_DEV_DBG, state); uart_print_endl(UART_DEV_DBG);
    uart_print(UART_DEV_DBG, "err_code="); uart_print_u8(UART_DEV_DBG, err_code); uart_print_endl(UART_DEV_DBG);
    uart_print(UART_DEV_DBG, "fs_st="); uart_print_u8(UART_DEV_DBG, fs_st); uart_print_endl(UART_DEV_DBG);
    uart_print(UART_DEV_DBG, "fs_err="); uart_print_u8(UART_DEV_DBG, fs_err); uart_print_endl(UART_DEV_DBG);
}

void dcmd_read_dbg_uart(void)
{
    uint8_t i, j, k;

    msg_buf_len[UART_DEV_DBG] += uart1_recv((uint8_t*)msg_buf[UART_DEV_DBG] + msg_buf_len[UART_DEV_DBG],
        DCMD_BUF_SIZE - msg_buf_len[UART_DEV_DBG]);
    if(msg_buf_len[UART_DEV_DBG] > 0)
    {
        for(i = 0; i < msg_buf_len[UART_DEV_DBG]; i++)
        {
            if(msg_buf[UART_DEV_DBG][i] == '\r')
            {
                msg_buf[UART_DEV_DBG][i] = 0;
                break;
            }
        }
        if(i < msg_buf_len[UART_DEV_DBG])
        {
            if((k = strcontain(msg_buf[UART_DEV_DBG], "unit")) > 0)
            {
                dcmd_unit(msg_buf[UART_DEV_DBG] + k);
            }
            else if(unit < UNIT_NUM)
            {
                // bridge to command uart
                uart2_send((uint8_t*)msg_buf[UART_DEV_DBG], i, 1);
                uart_putc(UART_DEV_CMD, '\r');
            }
            else
            {
                for(j = 0; j < CMD_COUNT; j++)
                {
                    dbg_cmd_t* cmd = &cmd_list[j];
                    if((k = strcontain(msg_buf[UART_DEV_DBG], cmd->cmd)) > 0)
                    {
                        cmd_list[j].exec(msg_buf[UART_DEV_DBG] + k);
                        break;
                    }
                }
                if(j >= CMD_COUNT)
                {
                    uart_print(UART_DEV_DBG, "ng unknown, \"");
                    uart_print(UART_DEV_DBG, msg_buf[UART_DEV_DBG]);
                    uart_print(UART_DEV_DBG, "\""); uart_print_endl(UART_DEV_DBG);
                }
            }

            // Erase the read message
            i++;
            for(j = 0; i < msg_buf_len[UART_DEV_DBG]; j++, i++)
            {
                msg_buf[UART_DEV_DBG][j] = msg_buf[UART_DEV_DBG][i];
            }
            msg_buf_len[UART_DEV_DBG] = j;
        }
    }

}

void dcmd_read_cmd_uart(void)
{
    uint8_t i, j;

    msg_buf_len[UART_DEV_CMD] += uart2_recv((uint8_t*)msg_buf[UART_DEV_CMD] + msg_buf_len[UART_DEV_CMD],
        DCMD_BUF_SIZE - msg_buf_len[UART_DEV_CMD]);
    if(msg_buf_len[UART_DEV_CMD] > 0)
    {
        for(i = 0; i < msg_buf_len[UART_DEV_CMD]; i++)
        {
            if(msg_buf[UART_DEV_CMD][i] == '\n')
            {
                break;
            }
        }
        if(i < msg_buf_len[UART_DEV_CMD])
        {
            // bridge to debug uart
            i++;
            uart1_send((uint8_t*)msg_buf[UART_DEV_CMD], i, 1);

            // Erase the read message
            for(j = 0; i < msg_buf_len[UART_DEV_CMD]; j++, i++)
            {
                msg_buf[UART_DEV_CMD][j] = msg_buf[UART_DEV_CMD][i];
            }
            msg_buf_len[UART_DEV_CMD] = j;
        }
    }
}

void dcmd_execute(void)
{
    dcmd_read_dbg_uart();
    dcmd_read_cmd_uart();
}
