// Copyright 2023 Tomoyuki Watanabe
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>
#include "common.h"
#include "chgv.h"
#include "uart1.h"
#include "uart2.h"
#include "utility.h"
#include "uart_util.h"
#include "tmr.h"
#include "fsys.h"

#define DEBUG_PRINT_ENABLE 1

#if DEBUG_PRINT_ENABLE
#define PRINT_DEBUG(...) uart_print(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_ENDL() uart_print_endl(UART_DEV_DBG)
#define PRINT_DEBUG_U8(...) uart_print_u8(UART_DEV_DBG, __VA_ARGS__)
#define PRINT_DEBUG_I8(...) uart_print_i8(UART_DEV_DBG, __VA_ARGS__)
#else
#define PRINT_DEBUG(...) {}
#define PRINT_DEBUG_ENDL() {}
#define PRINT_DEBUG_U8(...) {}
#define PRINT_DEBUG_I8(...) {}
#endif

#define CHGV_CMD_TIMEOUT      500 // msec
#define CHGV_SAVE_CMD_TIMEOUT 2000 // msec
#define CHGV_CMD_RXBUF_SIZE   64
#define CHGV_CMD_RETRY_COUNT  3
char chgv_cmd_rxbuf[CHGV_CMD_RXBUF_SIZE];

chgv_status_t chgv_status[CHGV_MAX_NOF_SLOTS];
uint16_t chgv_bat_cap[CHGV_MAX_NOF_SLOTS];
__uint24 chgv_n_chg_time[CHGV_MAX_NOF_SLOTS];

void chgv_set_com_slot(uint8_t slot)
{
    switch(slot)
    {
        case 0:
            LATB = (LATB & 0xcf);
            break;
        case 1:
            LATB = ((LATB & 0xcf) | 0x20);
            break;
        case 2:
            LATB = ((LATB & 0xcf) | 0x10);
            break;
        default:
            LATB = ((LATB & 0xcf) | 0x30);
            break;
    }
}

// This function reads the response from charger unit.
// If ok response was received, it returns 0.
// If ng response was received, it returns 1.
// If st response was received, it returns 2.
// If recevie timeout occurred, ti returns -1.
// param_pos is set the parameter start index of chgv_cmd_rxbuf.
int8_t chgv_read_response_line(uint8_t* param_pos, uint32_t timeout)
{
    uint32_t start_tick;
    uint8_t i, j;
    int8_t rv = -1;
    uint8_t len = 0;
    start_tick = tmr_get_tick();
    while((tmr_get_tick() - start_tick) < timeout)
    {
        len += uart2_recv((uint8_t*)chgv_cmd_rxbuf + len,
                CHGV_CMD_RXBUF_SIZE - len);
        for(i = 0; i < len; i++)
        {
            if(chgv_cmd_rxbuf[i] == '\n')
            {
                chgv_cmd_rxbuf[i + 1] = 0;
                break;
            }
        }
        if(i < len)
        {
            j = strcontain(chgv_cmd_rxbuf, "ok");
            if(j > 0)
            {
                rv = 0;
                goto exit;
            }
            j = strcontain(chgv_cmd_rxbuf, "ng");
            if(j > 0)
            {
                rv = 1;
                goto exit;
            }

            PRINT_DEBUG("invalid res, ");
            PRINT_DEBUG(chgv_cmd_rxbuf);
            // Erase the read message
            i++;
            for(j = 0; i < len; j++, i++)
            {
                chgv_cmd_rxbuf[j] = chgv_cmd_rxbuf[i];
            }
            len = j;
        }
    }
exit:
    PRINT_DEBUG("rv="); PRINT_DEBUG_I8(rv); PRINT_DEBUG_ENDL();
    if(rv >= 0) *param_pos = j;
    return rv;
}

int8_t chgv_read_config_u8(uint8_t* value, const char* var_name)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) < 0)
    {
        return -1;
    }
    if(stou8(value, chgv_cmd_rxbuf + i) == 0)
    {
        return -2;
    }

    return 0;
}

int8_t chgv_read_config_u16(uint16_t* value, const char* var_name)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) < 0)
    {
        return -1;
    }
    if(stou16(value, chgv_cmd_rxbuf + i) == 0)
    {
        return -2;
    }

    return 0;
}

int8_t chgv_read_config_u24(__uint24* value, const char* var_name)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) < 0)
    {
        return -1;
    }
    if(stou24(value, chgv_cmd_rxbuf + i) == 0)
    {
        return -2;
    }

    return 0;
}

int8_t chgv_write_config_u8(const char* var_name, uint8_t value)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, ' ');
    uart_print_u8(UART_DEV_CMD, value); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) != 0)
    {
        return -1;
    }

    return 0;
}

int8_t chgv_write_config_u16(const char* var_name, uint16_t value)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, ' ');
    uart_print_u16(UART_DEV_CMD, value); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) != 0)
    {
        return -1;
    }

    return 0;
}

int8_t chgv_write_config_u24(const char* var_name, __uint24 value)
{
    uint8_t i;
    uart_print(UART_DEV_CMD, var_name); uart_putc(UART_DEV_CMD, ' ');
    uart_print_u24(UART_DEV_CMD, value); uart_putc(UART_DEV_CMD, '\r');
    if(chgv_read_response_line(&i, CHGV_CMD_TIMEOUT) != 0)
    {
        return -1;
    }

    return 0;
}

int8_t chgv_read_config(uint8_t slot)
{
    uint8_t i;
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    
    PRINT_DEBUG("slot"); PRINT_DEBUG_U8(slot); PRINT_DEBUG_ENDL(); 
    for(i = 0; i < CHGV_CMD_RETRY_COUNT; i++)
    {
        PRINT_DEBUG("batcap"); PRINT_DEBUG_ENDL(); 
        if(chgv_read_config_u16(&chgv_bat_cap[slot], "batcap") != 0)
        {
            continue;
        }
        PRINT_DEBUG("nchgtime"); PRINT_DEBUG_ENDL(); 
        if(chgv_read_config_u24(&chgv_n_chg_time[slot], "nchgtime") != 0)
        {
            continue;
        }
        break;
    }

    return (i < CHGV_CMD_RETRY_COUNT) ? 0: -1;
}

int8_t chgv_write_config(uint8_t slot)
{
    uint8_t i;
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    
    PRINT_DEBUG("slot"); PRINT_DEBUG_U8(slot); PRINT_DEBUG_ENDL(); 
    for(i = 0; i < CHGV_CMD_RETRY_COUNT; i++)
    {
        PRINT_DEBUG("batcap"); PRINT_DEBUG_ENDL(); 
        if(chgv_write_config_u16("batcap", chgv_bat_cap[slot]) != 0)
        {
            continue;
        }
        break;
    }

    return (i < CHGV_CMD_RETRY_COUNT) ? 0: -1;
}

int8_t chgv_send_no_param_cmd(uint8_t slot, const char* cmd)
{
    uint8_t i;
    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();

    PRINT_DEBUG("slot"); PRINT_DEBUG_U8(slot); PRINT_DEBUG_ENDL(); 
    for(i = 0; i < CHGV_CMD_RETRY_COUNT; i++)
    {
        uint8_t res_pos;
        uart_print(UART_DEV_CMD, cmd); uart_putc(UART_DEV_CMD, '\r');
        PRINT_DEBUG(cmd); PRINT_DEBUG_ENDL(); 
        if(chgv_read_response_line(&res_pos, CHGV_SAVE_CMD_TIMEOUT) != 0)
        {
            continue;
        }
        break;
    }

    return (i < CHGV_CMD_RETRY_COUNT) ? 0: -1;
}

int8_t chgv_read_status(uint8_t slot, void (*read_callback)(uint8_t slot, char* status_str))
{
    uint8_t i;
    int8_t rv;
    uint8_t status_count = 0;

    PRINT_DEBUG("slot"); PRINT_DEBUG_U8(slot); PRINT_DEBUG_ENDL(); 

    chgv_set_com_slot(slot);
    uart2_flush_rx_buf();
    for(i = 0; i < CHGV_CMD_RETRY_COUNT;)
    {
        uint8_t res_pos;
        PRINT_DEBUG("status"); PRINT_DEBUG_ENDL(); 
        uart_print(UART_DEV_CMD, "status\r");
        rv = chgv_read_response_line(&res_pos, CHGV_CMD_TIMEOUT);
        if(rv == 0)
        {
            read_callback(slot, chgv_cmd_rxbuf + res_pos);
        }
        else if(rv == 1)
        {
            break;
        }
        else
        {
            i++;
            continue;
        }
    }

    return (status_count == 0) ? -1: 0;
}

int8_t chgv_save_config(uint8_t slot)
{
    return chgv_send_no_param_cmd(slot, "savecfg");
}

int8_t chgv_start(uint8_t slot)
{
    return chgv_send_no_param_cmd(slot, "start");
}

int8_t chgv_reset(uint8_t slot)
{
    return chgv_send_no_param_cmd(slot, "reset");
}

const char chgv_st_char[CHGV_ST_NUM] =
{
    'I', // CHGV_ST_INIT           0
    'I', // CHGV_ST_INIT_MES       1
    'I', // CHGV_ST_CHK            2
    'D', // CHGV_ST_DSC            3
    'D', // CHGV_ST_DSC_MES        4
    'N', // CHGV_ST_N_CHG          5
    'N', // CHGV_ST_N_VOL_MES      6
    'Q', // CHGV_ST_Q_CHG          7
    'Q', // CHGV_ST_Q_VOL_MES      8
    'F', // CHGV_ST_FIN            9
    'F', // CHGV_ST_FIN_MES       10
    'E', // CHGV_ST_ERR           11
    'N', // CHGV_ST_N_CHG_VOL_TST 12
    'Q', // CHGV_ST_Q_CHG_VOL_TST 13
    'N', // CHGV_ST_N_CHG_TST     14
    'N', // CHGV_ST_N_CHG_TST_MES 15
    'Q', // CHGV_ST_Q_CHG_TST     16
    'Q', // CHGV_ST_Q_CHG_TST_MES 17
    'D', // CHGV_ST_DSC_TST       18
    'O', // CHGV_ST_OFF_CAL       19
    'M'  // CHGV_ST_CHG_MES_TST   20
};

const char* chgv_err_text(uint8_t err_code)
{
    // up to 4 characters
    switch(err_code)
    {
        case CHGV_ERR_NONE:
            return "NONE";
        case CHGV_ERR_BAT_MIN_VOL:
            return "MINV";
        case CHGV_ERR_SHORT:
            return "SHRT";
        case CHGV_ERR_OPEN:
            return "OPEN";
        case CHGV_ERR_CHG_DUR:
            return "DUR";
        case CHGV_ERR_OVER_CUR:
            return "OCUR";
        default:
            return "UNKN";
    }
}

int8_t chgv_set_status(uint8_t slot, const char* status_str)
{
    chgv_status_t status;
    uint8_t i, k;
    int8_t rv = 0;

    i = 0;
    k = stou32(&status.tick, status_str);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou8(&status.prev_state, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou8(&status.state, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou8(&status.err_code, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou16(&status.open_bat_vol, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou16(&status.chgdsc_bat_vol, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou16(&status.chgdsc_current, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    i += k;
    k = stou16(&status.prev_chgdsc_bat_vol, status_str + i);
    if(k == 0)
    {
        rv = -1;
        goto end;
    }
    memcpy(&chgv_status[slot], &status, sizeof(chgv_status_t));

end:
    return rv;
}
