﻿using System;
using System.Reflection;
using System.Runtime.CompilerServices;


namespace SDR
{
    class Osc
    {
        double phase = 0;   // 位相値
        //
        //  任意の周波数の信号を作成する(負の周波数にも対応)
        //
        public void CreateSignal(short[] real, short[] imag, double freq, double level)
        {
            for (int i = 0; i < 1024; i++)
            {
                real[i] = F16Math.ToF16(Math.Cos(phase) * level);  // X軸の座標を生成
                imag[i] = F16Math.ToF16(Math.Sin(phase) * level); // Y軸の座標を生成
                phase += freq / 8000 * Math.PI;  // 位相を進める
                if (phase > 2 * Math.PI)  // 一周したら
                    phase -= 2 * Math.PI;    // 必ず0から2*PIの範囲に収めておく
            }
        }
    }
    class Signal
    {
        Random rnd = new Random();  // 乱数クラスのオブジェクトを準備する
        short[] temp = new short[1024];   // 交換用作業領域確保
        short lastphase = 0;
        int angle;
        //
        //  I/Qの入れ替え
        //
        public void SwapData(short[] real, short[] imag)
        {
            Array.Copy(real, temp, 1024);  //  まずrealを保存
            Array.Copy(imag, real, 1024);  //  imagをrealに上書き
            Array.Copy(temp, imag, 1024);  //  保存していたrealをimagにコピー
        }
        //
        //  信号バッファをクリアする
        //
        public void ClearSignal(short[] real, short[] imag)
        {
            Array.Clear(real, 0, 1024); // 配列をクリアする
            Array.Clear(imag, 0, 1024); // 配列をクリアする
        }
        //
        //  ２つの信号を足し算して合成する
        //
        public void AddSignal(short[] real, short[] imag, short[] localreal, short[] localimag)
        {
            for (int i = 0; i < 1024; i++)
            {
                real[i] += localreal[i];    // 加算して結果を格納
                imag[i] += localimag[i];    // 加算して結果を格納
            }
        }
        //
        //  ２つの信号を掛け算して周波数変換を行なう
        //
        public void MultiplySignal(short[] real, short[] imag, short[] localreal, short[] localimag)
        {
            for (int i = 0; i < 1024; i++)
            {
                F16Math.CompMul(real[i], imag[i], localreal[i], localimag[i], out real[i], out imag[i]);
            }
        }
        //
        //  信号の絶対値(原点からの距離)を求めてその最大値を返す
        //
        public void AmDemod(short[] real, short[] imag)
        {
            for (int i = 0; i < 1024; i++)
            {
                short norm = F16Math.Norm(real[i], imag[i]);
                real[i] = imag[i] = norm;
            }
        }
        
        //
        //  信号の位相(real軸からの角度)を求める
        //
        public void GetPhase(short[] result, short[] real, short[] imag)
        {
            for (int i = 0; i < 1024; i++)
            {
                result[i] = F16Math.Atan2(real[i], imag[i]);  // ArcTanを求める
            }
        }
        //
        //  信号の周波数成分を求める
        //
        public void GetDivision(short[] phase)
        {
            int div;
            for (int i = 0; i < 1024; i++)
            {
                div = (int)phase[i] - (int)lastphase;
                lastphase = phase[i];
                if (div > 32767)
                {
                    div -= 32768 * 2;
                }
                else if (div < -32786)
                {
                    div += 2 * 32768;
                }                    
                phase[i] = (short)div;
            }
        }
        //
        //  信号に定数を掛けて増幅/減衰する
        //
        public void MulConst(short[] real, short[] imag, short gain)
        {
            for (int i = 0; i < 1024; i++)
            {
                real[i] = F16Math.Mul(real[i], gain);
                imag[i] = F16Math.Mul(imag[i], gain);
            }
        }
        //
        //  信号に変調信号を掛けて振幅変調する
        //
        public void AmMod(short[] real, short[] imag, short[] af)
        {
            for (int i = 0; i < 1024; i++)
            {
                real[i] = F16Math.Mul(real[i], (short)(af[i] >> 1));
                imag[i] = F16Math.Mul(imag[i], (short)(af[i] >> 1));
            }
        }
        //
        //  信号を変調信号に従って周波数変調する
        //
        public void FmMod(short[] real, short[] imag, short[] af)
        {
            for (int i = 0; i < 1024; i++)
            {
                // 積分して位相変調すれば周波数変調になる
                angle += af[i];
                if (angle > 32767)
                {
                    angle -= 32768 * 2;
                }
                else if (angle < -32786)
                {
                    angle += 2 * 32768;
                }
                short modreal = F16Math.ToF16(Math.Cos((double)angle / 32768.0) / 32768.0);
                short modimag = F16Math.ToF16(Math.Sin((double)angle / 32768.0) / 32768.0);
                short a = (short)(F16Math.Mul(real[i], modreal) - F16Math.Mul(imag[i], modimag));
                short b = (short)(F16Math.Mul(real[i], modimag) + F16Math.Mul(imag[i], modreal));
                real[i] = a;
                imag[i] = b;
            }
        }

        //
        //  テスト用にホワイトノイズを発生させる
        //
        public void CreateNoise(short[] real, short[] imag, double level)
        {
            for (int i = 0; i < 1024; i++)
            {
                double angle = (rnd.Next(2 * 32768 + 1) - 32768) * 2.0 * Math.PI / 32768;
                real[i] = F16Math.ToF16(Math.Cos(angle) * level);
                imag[i] = F16Math.ToF16(Math.Sin(angle) * level);
            }
        }
    }

    class Agc
    {
        int applyTimeBits = 3; // 2 ^ 3 = 8
        int sampleMaxCount = 16000; // 1sec

        int sampleCount = 0;
        short max = 0;
        int accGain = 0;
        int gainShift;
        short gainFrac;
        const short targetAmp = 0x3fff;

        public int GainShift
        {
            get
            {
                return gainShift;
            }
        }
        public short GainFrac
        {
            get
            {
                return gainFrac;
            }
        }

        public void SetParam(int sampleMaxCount, int applyTime)
        {
            this.sampleMaxCount = sampleMaxCount;
            this.applyTimeBits = 0;
            for (this.applyTimeBits = 12; this.applyTimeBits > 0; this.applyTimeBits--)
            {
                if ((applyTime >> this.applyTimeBits) == 1)
                    break;
            }
            sampleCount = 0;
            max = 0;
            accGain = 1 << this.applyTimeBits;
            gainShift = 0;
            gainFrac = 0;
        }

        public void Reset()
        {
            applyTimeBits = 0;
            sampleCount = 0;
            max = 0;
            accGain = 1 << applyTimeBits;
            gainShift = 0;
            gainFrac = 0;
        }
        public void GetParam(out int sampleMaxCount, out int applyTime)
        {
            sampleMaxCount = this.sampleMaxCount;
            applyTime = 1 << this.applyTimeBits;
        }

        private int FindBitChangeLeft(short v)
        {
            int sign = (v & 0x8000);
            int shift = 0;
            v <<= 1;
            while(shift < 15 && (v & 0x8000) == sign)
            {
                v <<= 1;
                shift++;
            }
            return -shift;
        }
        private int FindFirstOneFromLeft(short v)
        {
            int maxShift = 17;
            int sign = (v & 0x8000);
            int shift = 1;
            while (shift < maxShift && (v & 0x8000) == 0)
            {
                v <<= 1;
                shift++;
            }
            return (shift == maxShift) ? 0: shift;
        }

        public void CalcAgcGain(short[] norm)
        {
            short curGain = (short)((1 << gainShift) + gainFrac);

            for (int i = 0; i < 1024; i++)
            {
                short norm0 = (short)(norm[i] / curGain);
                if (max < norm0)
                {
                    max = norm0;
                }
                
                if (++sampleCount >= sampleMaxCount)
                {
                    if (max > 0)
                    {
                        short gain = (short)(targetAmp / max);
                        accGain = accGain - (accGain >> applyTimeBits) + gain;
                        gain = (short)(accGain >> applyTimeBits);

                        gainShift = 16 - FindFirstOneFromLeft(gain);
                        gainFrac = (short)(gain - (1 << gainShift));
                    }
                    sampleCount = 0;
                    max = 0;
                }
            }
        }
    }
}
