﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace SDR
{
    class Filter
    {
        int taps;    // TAP数
        int histptr;    // 過去の信号の先頭インデックス
        double[] fircoef0;   // FIRフィルタ係数バッファ
        int scaleBits;
        short[] fircoef;   // FIRフィルタ係数バッファ
        short[] realhistory;   // 過去の信号の保存バッファ(実部)
        short[] imaghistory;   // 過去の信号の保存バッファ(虚部)
        double lowcut;
        double highcut;
        //
        //  Sinc関数 Sin(X)/X
        //
        public double Sinc(double x)
        {
            if (Math.Abs(x) < 1e-10) // sinc(0) = 1の実装
                return 1.0;
            else
                return Math.Sin(Math.PI * x) / (Math.PI * x); // Sinc(x) = Sin(Math.PI*x)/(Math.PI*X)
        }
        //
        //  ハミング窓関数
        //
        public double Hamming(double x)
        {
            return 0.54 - 0.46 * Math.Cos(2 * Math.PI * x);  // ハミング窓関数を得る
        }
        //
        //  FIRフィルタの初期化を行なう
        //
        public void CreateFirFilter(int taps, double lowcut, double highcut)
        {
            double x, h;
            this.lowcut = lowcut;
            this.highcut = highcut;

            fircoef0 = new double[taps];  // FIR係数を保持するバッファ
            fircoef = new short[taps];  // FIR係数を保持するバッファ
            realhistory = new short[taps];  // 過去の実数信号を保存するバッファ
            imaghistory = new short[taps];  // 過去の虚数信号を保存するバッファ
            histptr = 0;    // 過去の信号の先頭を示すインデックス
            this.taps = taps; // タップ数を保存する
            double max = 0;
            for (int i = 0; i < taps; i++)   // フィルタ係数の生成ループ
            {
                x = i - (taps - 1.0) / 2.0;      // Sinc関数用の正規化値(-TAP/2からTAP/2までの範囲)
                h = i * (1.0 / (taps - 1.0));    // ハミング窓関数計算用の正規化値(0から1までの範囲)
                fircoef0[i] = 2 * highcut * Sinc(2 * highcut * x) - 2 * lowcut * Sinc(2 * lowcut * x);// FIR係数を算出
                fircoef0[i] *= Hamming(h);  // 生成した係数に窓関数を掛ける
                if (max < Math.Abs(fircoef0[i])) max = fircoef0[i];
            }
            double gain = 1.0 / max;
            scaleBits = (int)(Math.Log(gain) / Math.Log(2));
            //if (scaleBits > 7) scaleBits = 7;
            gain = (1 << scaleBits);
            for (int i = 0; i < taps; i++)
            {
                fircoef[i] = F16Math.ToF16(fircoef0[i] * gain);
            }
        }
        public void GetParam(out int taps, out double lowcut, out double highcut)
        {
            taps = this.taps;
            lowcut = this.lowcut;
            highcut = this.highcut;
        }

        //
        //  複素信号にFIRフィルタを適用する
        //
        public void FirFilter(short[] real, short[] imag, int shift, short frac)
        {
            for (int i = 0; i < 1024; i++)
            {
                long realsum = 0;
                long imagsum = 0;
                for (int j = 0; j < taps; j++)
                {
                    realsum += (long)realhistory[histptr] * fircoef[j];
                    imagsum += (long)imaghistory[histptr] * fircoef[j];
                    if (++histptr == taps)
                        histptr = 0;
                }
                realhistory[histptr] = real[i];
                imaghistory[histptr] = imag[i];
                if (++histptr == taps)
                    histptr = 0;
                real[i] = (short)(realsum >> (15 + scaleBits - shift));
                real[i] += (short)((real[i] * frac) >> shift);
                imag[i] = (short)(imagsum >> (15 + scaleBits - shift));
                imag[i] += (short)((imag[i] * frac) >> shift);
            }
        }

        public void DumpCoef(string filename)
        {
            using (StreamWriter sw = new StreamWriter(filename))
            {
                sw.WriteLine(string.Format("scaleBits={0:d}", scaleBits));
                for (int i = 0; i < taps; i++)
                {
                    sw.WriteLine(string.Format("{0:X4}", fircoef[i]));
                }
            }
        }
    }
}
