﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace SDR
{
    class Filter
    {
        double PI = 3.141592653589793238;   // パイの値
        int taps;    // TAP数
        int histptr;    // 過去の信号の先頭インデックス
        double[] fircoef;   // FIRフィルタ係数バッファ
        double[] realhistory;   // 過去の信号の保存バッファ(実部)
        double[] imaghistory;   // 過去の信号の保存バッファ(虚部)
        double lowcut;
        double highcut;
        //
        //  Sinc関数 Sin(X)/X
        //
        public double Sinc(double x)
        {
            if (Math.Abs(x) < 1e-10) // sinc(0) = 1の実装
                return 1.0;
            else
                return Math.Sin(PI * x) / (PI * x);// Sinc(x) = Sin(PI*x)/(PI*X)
        }
        //
        //  ハミング窓関数
        //
        public double Hamming(double x)
        {
            return 0.54 - 0.46 * Math.Cos(2 * PI * x);  // ハミング窓関数を得る
        }
        //
        //  FIRフィルタの初期化を行なう
        //
        public void CreateFirFilter(int taps, double lowcut, double highcut)
        {
            double x, h;
            this.lowcut = lowcut;
            this.highcut = highcut;

            fircoef = new double[taps];  // FIR係数を保持するバッファ
            realhistory = new double[taps];  // 過去の実数信号を保存するバッファ
            imaghistory = new double[taps];  // 過去の虚数信号を保存するバッファ
            histptr = 0;    // 過去の信号の先頭を示すインデックス
            this.taps = taps; // タップ数を保存する
            for (int i = 0; i < taps; i++)   // フィルタ係数の生成ループ
            {
                x = i - (taps - 1.0) / 2.0;      // Sinc関数用の正規化値(-TAP/2からTAP/2までの範囲)
                h = i * (1.0 / (taps - 1.0));    // ハミング窓関数計算用の正規化値(0から1までの範囲)
                fircoef[i] = 2 * highcut * Sinc(2 * highcut * x) - 2 * lowcut * Sinc(2 * lowcut * x);// FIR係数を算出
                fircoef[i] *= Hamming(h);  // 生成した係数に窓関数を掛ける
            }
        }
        public void GetParam(out int taps, out double lowcut, out double highcut)
        {
            taps = this.taps;
            lowcut = this.lowcut;
            highcut = this.highcut;
        }

        //
        //  複素信号にFIRフィルタを適用する
        //
        public void FirFilter(double[] real, double[] imag)
        {
            double realsum, imagsum;
            for (int i = 0; i < 1024; i++)   // 1024サンプル
            {
                realsum = 0;    // 合計をクリア
                imagsum = 0;    // 合計をクリア
                for (int j = 0; j < taps; j++)   // TAP数だけ繰り返す
                {
                    realsum += realhistory[histptr] * fircoef[j];    // 信号に係数を掛けて積算していく
                    imagsum += imaghistory[histptr] * fircoef[j];    // 信号に係数を掛けて積算していく
                    if (++histptr == taps)  // 終端に到達？
                        histptr = 0;    // 履歴インデックスを最初に戻す
                }
                realhistory[histptr] = real[i]; // 履歴を保存
                imaghistory[histptr] = imag[i]; // 履歴を保存
                if (++histptr == taps)  // 終端に到達？
                    histptr = 0;    // 履歴インデックスを最初に戻す
                real[i] = realsum;  // フィルタを適用した後のデータを格納
                imag[i] = imagsum;  // フィルタを適用した後のデータを格納
            }
        }

        public void DumpCoef(string filename)
        {
            double max = 0;
            for (int i = 0; i < taps; i++)
            {
                if (max < fircoef[i]) max = fircoef[i];
            }
            int scale = (int)(1.0 / max);
            int scale_bit = 30;
            for (; scale_bit > 0; scale_bit--)
            {
                if (((1 << scale_bit) & scale) != 0)
                    break;
            }
            scale = (1 << scale_bit);

            using (StreamWriter sw = new StreamWriter(filename))
            {
                sw.WriteLine(string.Format("max={0:g}", max));
                sw.WriteLine(string.Format("scale={0:d}", scale));
                for (int i = 0; i < taps; i++)
                {
                    sw.WriteLine(string.Format("{0:g}", fircoef[i]));
                }
                for (int i = 0; i < taps; i++)
                {
                    sw.WriteLine(string.Format("{0:X4}", (short)(fircoef[i] * scale * 0x7fff)));
                }
            }
        }
    }
}
