﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SDR
{
    class Fft
    {
        double[] real;   // FFTの結果
        double[] imag;   // FFTの結果
        double PI = 3.141592653589793238;   // パイの値
        double[] window;    // 窓関数用の係数バッファ
        int fftbin;    // FFTのBIN数の保存用
        int[] revindex; // バタフライ演算用の逆インデックステーブル
        //
        //  FFTの初期化(コンストラクタ)
        //
        public Fft(int size, int func)
        {
            real = new double[size]; // FFTするデータ領域を確保(実部)
            imag = new double[size]; // FFTするデータ領域を確保(虚部)
            revindex = new int[size];   // バラフライインデックステーブル変数を準備
            int dist = size >> 1;   // バタフライインデックス距離を初期化
            fftbin = size;             // FFTのBIN数を保存
            for(int i = 1; i < size; i *= 2)   // バタフライインデックステーブルを作成
            {
                for(int j = 0; j < i; j++)     // 交換するインデックスを生成して(ビットリバース)
                    revindex[j + i] = revindex[j] + dist;   // 交換したインデックスを格納
                dist >>= 1;     // インデックス距離を半分にする
            }
            window = new double[fftbin];   // ウインドウ関数のバッファを生成
            SetupWindow(func);  // 窓関数を作成
        }
        //
        //  窓関数係数の作成
        //
        public void SetupWindow(int func)   // 窓関数を変更する際に呼ばれる
        {
            for(int i = 0; i < fftbin; i++)      // ウインドウ関数を生成
                switch(func)   // 窓関数の種類により分岐
                {
                case 0: // 矩形窓(窓なしと等価)
                    window[i] = 1;
                    break;
                case 1: // バートレット窓(三角窓)
                    window[i] = 1 - 2 * System.Math.Abs(((float)i / fftbin) - 0.5);
                    break;
                case 2: // ブラックマン窓 
                    window[i] = 0.42 - 0.5 * Math.Cos(2 * PI * i / fftbin) + 0.08 * Math.Cos(4 * PI * i / fftbin);
                    break;
                case 3: // ハミング窓 
                    window[i] = 0.54 - 0.46 * Math.Cos(2 * PI * i / fftbin);
                    break;
                case 4: // ハン窓 
                    window[i] = 0.5 - 0.5 * Math.Cos(2 * PI * i / fftbin);
                    break;
                }
        }
        //
        //  窓関数の適用
        //
        public void ApplyWindow(double[] realsrc, double[] imagsrc)
        {
            for(int i = 0; i < fftbin; i++) // FFTのbin数だけ繰り返す
            {
                realsrc[i] *= window[i];    // 窓関数を適用する
                imagsrc[i] *= window[i];    // 窓関数を適用する
            }
        }
        //
        //  両方向のFFTを計算する
        //
        public void GetFft(double[] realsrc, double[] imagsrc, bool rev)
        {
            int i, j, k, stage, type, dist, size;
            double twRe, twIm, rotRe, rotIm, tempRe, tempIm, div;

            for(i = 0; i < fftbin; i++)
            {
                real[i] = realsrc[revindex[i]];// 効率の良い演算のため場所を入れ替える
                imag[i] = imagsrc[revindex[i]];// 効率の良い演算のため場所を入れ替える
            }

            for(stage = 1; ; stage++)   // ステージは1,2,3,4と大きくなっていく、1024ビンなら10まで
            {
                dist = 1 << stage;  // バタフライ距離を算出(2,4,8,16,,,)
                if(dist > fftbin)  // 終了判定
                    break;  // stageが11になったらFFT計算ループを抜ける
                size = dist >> 1;  // ペアになる相手へのインデックス距離を算出(1,2,4,8...)

                twRe = 1.0;  // 回転因子(twidle factor)を0度に初期化
                twIm = 0.0;  // 回転因子を0度に初期化
                rotRe = Math.Cos(PI / size); // 回転因子の回転変化分を予め算出(FFTなら回転方向は負)
                rotIm = Math.Sin(PI / size) * (rev? 1.0: -1.0); // 回転因子の回転変化分を予め算出(逆FFTなら回転方向は正)

                for(type = 0; type < size; type++) // stage1なら0のみ、stage1なら0,1、stage2なら0,1,2,3と範囲が倍になっていく
                {
                    for(j = type; j < fftbin; j += dist)    // jはstage1なら0,2,4,6、stage2なら0,4,8,12と間隔が倍になっていく
                    {
                        k = j + size;  // バタフライ演算の相手先のインデックスを得る
                        tempRe = real[k] * twRe - imag[k] * twIm;// 回転因子を掛けて相手先を回転させる
                        tempIm = real[k] * twIm + imag[k] * twRe;// 回転因子を掛けて相手先を回転させる
                        real[k] = real[j] - tempRe;// 回転させた結果と元の値との差分を相手先に格納
                        imag[k] = imag[j] - tempIm;// 回転させた結果と元の値との差分を相手先に格納
                        real[j] += tempRe;// 元には回転させた結果を積分していく
                        imag[j] += tempIm;// 元には回転させた結果を積分していく
                    }
                    tempRe = twRe * rotRe - twIm * rotIm;   //回転因子を回転させる
                    tempIm = twRe * rotIm + twIm * rotRe;   //回転因子を回転させる
                    twRe = tempRe;  // 回転因子をアップデート
                    twIm = tempIm;  // 回転因子をアップデート
                }
            }
            div = rev? 1.0 / fftbin: 1.0;   // 順FFTならそのままだが逆FFTなら結果を1/Nにする
            for(i = 0; i < fftbin; i++) // 全データに対して行う
            {
                realsrc[i] = real[i] * div; // 必要な定数を掛けて計算結果を格納する
                imagsrc[i] = imag[i] * div; // 必要な定数を掛けて計算結果を格納する
            }
        }
        //
        //  FFTで得られた複素数から絶対値のdBに変換する(スペアナ表示用)
        //
        public void GetDecibel(double[] db, double[] real, double[] imag)
        {
            for(int i = 0; i < fftbin; i++) // 複素データを絶対値に変換しdB表示の為にLog10をとる
                db[i] = Math.Log10(Math.Sqrt(real[i] * real[i] + imag[i] * imag[i]) + 1e-5) * 20;// デシベル値を得るためLog10値を20倍する
        }
    
    }
}
